// Example4.h - Demonstrates use of LsDoor SDK's databases.

#ifndef __Example4_HEADER_FILE__   // Using these can be a good habit.
#define __Example4_HEADER_FILE__   // And prevents multiple-inclusion.

struct playerType {
  unsigned long num;               // Needed to work the database
  unsigned char bbshandle[20];     // Matches with user.handle
  unsigned char handle[20];        // Asked, not the same as user.handle
  unsigned short points;           // Game stuff, not used in this example
  unsigned short power;            // Game stuff, not used in this example
} player;


  // Indexing modes:     (Future versions of lsDoor may add more)
  //   (0)  Single Record - Do not use TData for this, see below...
  //   (1)  Standard Database - Uses Indexes and full database support...
  //   (2)  Independant - A complete database with no indexes...

  // Single Record:
  //   The easiest kind of data to work with is single record.  You needn't
  // create any TData objects for single record data files.  All you need to
  // do is setup a couple of macros like these:

/*
#define load_cfg() read_single("Example5.Cfg"), &cfg, sizeof(cfg), false )
#define save_cfg() write_single("Example5.Cfg"), &cfg, sizeof(cfg) )
*/

  // Standard Database:

  // The C++ Class TData is the heart of all the databases except
  // single-record.  The first step is to create an object of TData (in
  // this case we named it Player) using the file name, size of our data,
  // and the indexing mode we want.

  // TData::TData( char *filename, ushort datasize, uchar indexingMode );

#ifdef MainModule
TData Player( "Player.Dat", sizeof(player), 1 );
#else
extern TData Player;
#endif

  // The following macros are setup for use with an Indexed database.  The
  // key being used is 'player.bbshandle'.  To use a Independant database,
  // change the Indexing Mode above and then replace all occurances of
  // 'player.bbshandle' below with NULL.

  // You can have any number of databases, and any number of databases in
  // use at one time.  You don't have to create macros for them all, but
  // we think it will probably speed things up.

#define open_player()       Player.Open()
  // Opens the database.  Calls to this function may be nested as many times
  // as needed (i.e. you can call the function 10 times provided you also
  // call the close_player() function an equal number of times.)  No matter
  // what, the actual files will never be opened more than once.  Note that
  // this function has the side-effect of a reset_player() call, which
  // close_player() will NOT undo.  See below regarding reset_player().

#define next_player()       Player.Load( (void *)&player )
  // This loads the next record in the database.  Note that the order of
  // records is somewhat random.  This is the easiest and fastest way to
  // load data from a database except when using keys.  This function
  // will return false when it reaches the end of the database, otherwise
  // it will return true and fill the 'player' structure.

#define reset_player()      Player.Reset()
  // This restarts the next_player() position to the beginning of
  // the database.  Open_player() also causes a call to this function.

#define save_player()       Player.Save( player.bbshandle, player.num, (void *)&player )
  // This function saves the 'player' data to disk.  If the player.num is
  // already in use, the old record of the same number will be overwritten.
  // This allows you to modify a 'player' (modify anything except the
  // player.num) and replace your older data.  If the player.num does not
  // exist, the new record is added.  This function will also create (but
  // not modify) a key using (in this case) player.bbshandle ONLY if the
  // player.num does not already exist.

#define add_player()        player.num = Player.GetHigh(); \
                            Player.Add( player.bbshandle, player.num, (void *)&player )
  // This macro will save the current 'player' data to disk, and also assign
  // new player.num to the 'player' structure.  This is much faster than
  // save_player() but cannot overwrite previous data.  If indexing is
  // in use, this function will also create the new key, in this case, using
  // player.bbshandle.

#define delete_player()     Player.Delete( player.num, player.bbshandle )
  // To delete a record, load it into 'player', then call this function.
  // This erases a 'player' record on disk and the key that goes with it.

#define load_player(NUM)    Player.Load( NUM, (void *)&player )
  // This loads the player by their number.  Every record must have a
  // unique ID given by add_player().  This function returns false if the
  // player could not be found or on error.  This function is fairly slow.

#define end_player()        Player.end
  // This function will tell you if NUM of the last load_player() call
  // was past the end of the database.

#define close_player()      Player.Close()
  // Call to close an open database.  Calls to Open and Close may be nested
  // but require an equal number of calls to Open as to Close.  See also
  // open_player() above.


  ///////  Key related...

  // These functions let you load data using Indexes, a very fast method.
  // You must decide on what the key's will be, in this case we are using
  // the player's BBS handle (player.bbshandle) for the keys.

#define find_player(W)      Player.Load( (unsigned char *)W, (void *)&player )
  // This function uses the char string W to find a matching key.  The
  // search is not case sensitive.  If the key was found, the function
  // fills the 'player' structure with the matching record and returns true.

#define change_player(Old,New)  Player.Change( player.num, Old, New )
  // This function MODIFIES a key.  This does not CREATE a key.  The keys
  // are created by the save_player() and add_player() functions.
  // Old and New are both char strings.  You should load the 'player' data
  // being changed before calling.  An example where this might be used
  // would be if the user changed their handle on the BBS.  To be able to
  // find the key ever again you have to change it to match the new
  // user.handle.

  ///////  Rarely used and Obsolete functions...

#define version_player()    Player.CheckVersion()
  // This function most likely won't be needed by you.  It does not give the
  // version number of the database, but rather it gives the version of
  // TData which it is using.

#define isopen_player()     Player.isopen
  // Obsolete, but available.  Simply returns true/false whether the file
  // is open or not.

#define high_player()       Player.GetHigh()
  // This function gives the highest player.num currently in use.  Note that
  // this does not indicate how many records are in the database, but rather
  // it indicates how many records have been added since the database was
  // first created.

#define key_player()        Player.LoadKey()
  // This function is almost nothing but trouble.  It serves the purpose of
  // returning the next key in the indexes so that you can compare it
  // yourself.  When you find the key you want, you can call
  // thiskey_player() to fill the 'player' structure.

#define thiskey_player()    Player.LoadThisKey()  // Use with key_player()


#endif // End #ifndef __Example4_HEADER_FILE__


// End of Example4.h
