/* Uncomment the following line for multi-node compatible file access. */
#define MULTINODE_AWARE

/* Include standard C header files required by EZVote. */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <errno.h>
#include <assert.h>
#ifdef MULTINODE_AWARE
#include <io.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <share.h>
#endif

/* Common modules include files. */
#include "opendoor.h"

/* Include files specific to this project. */
#include "dispimb.h"
#include "title.h"
#include "main.h"
#include "header.h"
#include "chat.h"
#include "splash.h"
#include "pageview.h"


/* Manifest constants used by EZVote */
#define PROGRAM_NAME             "EZVote"
#define PROGRAM_VERSION          "Version 6.3"
#define PROGRAM_COPYRIGHT        "(C) Copyright 1991 - 1996 by Brian Pirie."
#define WELCOME_COPYRIGHT        "`bright red`         (C) Copyright 1991 - 1996 by Brian Pirie. All Rights Reserved."
#define WELCOME_NAME             "`bright red`                Welcome to EZVote 6.3 - The Online Poll System!\n\r\n\r"


#define NO_QUESTION              -1
#define NEW_ANSWER               -1
#define ALL_QUESTIONS            -2

#define QUESTIONS_VOTED_ON        0x0001
#define QUESTIONS_NOT_VOTED_ON    0x0002
#define QUESTIONS_AUTHORED        0x0004
#define QUESTIONS_DELETED         0x0008
#define QUESTIONS_NOT_DELETED     (QUESTIONS_VOTED_ON | QUESTIONS_NOT_VOTED_ON)
#define VOTE_ON_ALL               0x0010

#define MAX_QUESTIONS             400
#define MAX_USERS                 32000
#define MAX_ANSWERS               15
#define QUESTION_STR_SIZE         71
#define CATEGORY_STR_SIZE         31
#define ANSWER_STR_SIZE           31

#define GROW_CATEGORY_ARRAY_BY    10

#define USER_FILENAME             "EZVUSER.DAT"
#define QUESTION_FILENAME         "EZVQUEST.DAT"

#define TEMP_USER_FILENAME        "EZVUSER.TMP"
#define TEMP_QUESTION_FILENAME    "EZVQUEST.TMP"

#define FILE_ACCESS_MAX_WAIT      20

#define QUESTION_PAGE_SIZE        17

#define VERSION_NUM               0x0600


/* Use single-byte structure packing for file records. */
#ifdef __TURBOC__
#if(__TURBOC__ >= 0x295)
#pragma option -a-
#endif /* __TURBOC__ >= 0x295 */
#endif /* __TURBOC__ */
#ifdef _MSC_VER
#pragma pack(1)
#endif /* _MSC_VER */

/* Structure of records stored in the EZVUSER.DAT file */
typedef struct
{
   char szUserName[36];
   DWORD lUnused;
   WORD ausVotes[MAX_QUESTIONS];
} tUserRecord;

typedef struct
{
   WORD unVersionNum;
   time_t TimeInstalled;
   INT16 nMaxQuestions;
   DWORD lUnused;
   INT16 nHeaderChecksum;
} tUserFileHeader;
              


/* Structure of records stored in the EZVQUEST.DAT file */
typedef struct
{
   char szQuestion[QUESTION_STR_SIZE];
   char szCategory[CATEGORY_STR_SIZE];
   BYTE aszAnswer[MAX_ANSWERS][ANSWER_STR_SIZE];
   INT16 nTotalAnswers;
   WORD auVotesForAnswer[MAX_ANSWERS];
   WORD uTotalVotes;
   BYTE bCanAddAnswers;
   BYTE bMultipleAnswers;
   BYTE bAnonymous;
   BYTE bDeleted;
   char szCreatorName[36];
   time_t lCreationTime;
   DWORD lUnused;
} tQuestionRecord;

/* Restore original structure alignment, if possible. */
#ifdef _MSC_VER
#pragma pack()
#endif /* _MSC_VER */

/* Current User Information. */
tUserFileHeader UserFileHeader;
tUserRecord CurrentUserRecord;
int nCurrentUserNumber;


/* In-memory structures for category information. */
typedef struct
{
   char szCategory[CATEGORY_STR_SIZE];
   BOOL bAnyVotedInCategory;
   BOOL bAnyNotVotedInCategory;
   BOOL bAnyAuthoredInCategory;
   BOOL bAnyDeletedInCategory;
} tCategoryInfoRecord;

typedef struct
{
   int nTotalCategories;
   int nCategoryArraySize;
   tCategoryInfoRecord *paCategoryRecord;
} tCategoryInfo;

typedef int tQuestionPos;
#define ResetQuestionPos(x) (*(x) = 0)

/* Global variables. */
tCategoryInfo CategoryInfo = {0, 0, NULL};
int nViewResultsFrom = QUESTIONS_VOTED_ON;
char bAllowAdd = TRUE;
char bAllowChange = FALSE;
char bAllowUserDelete = FALSE;
char bAlwaysForceAnswer = FALSE;
BOOL bShowTitleScreen = TRUE;
BOOL bConfirmDoorExit = TRUE;
BOOL bShowSingleCategory = TRUE;
BOOL bAllowUserCategoryCreation = TRUE;
BOOL bUseCategories = TRUE;
int nQuestionsVotedOn = 0;
BOOL bTitleDisplayed = FALSE;
BOOL bUseOldUI = TRUE;
char szMainMenu[200];
int nVoteCmdIndex = 100;
int nChangeCmdIndex = 100;
int nAddCmdIndex = 100;
int nDeleteCmdIndex = 100;
int nViewCmdIndex = 100;
int nPageCmdIndex = 100;
int nQuitCmdIndex = 100;
int nGoodbyeCmdIndex = 100;
int nSysopCmdIndex = 100;
BYTE btWinBrdrColor = 0x70;
BYTE btWinTitleColor = 0x74;
BYTE btWinTextColor = 0x70;
char chYesOrNo;


/* Prototypes for functions that form EZVote */
void DrawMenuOption(int nCol, int nRow, char chCommandKey, char *pszOptionText);
void TitleScreen(void);
void CommandGoodbye(void);
void CommandQuit(void);
void ClearWithHeader(char *pszHeaderText);
char UserPrompt(char *pszHeaderText, char *pszMessageLine1,
   char *pszMessageLine2, char *pszPossibleKeys);
void MessageScreen(char *pszHeaderText, char *pszMessageLine1,
   char *pszMessageLine2);
void CustomConfigFunction(char *pszKeyword, char *pszOptions);
void BeforeExitFunction(void);
unsigned short HasVotedOnAnswer(unsigned short usVote, int nAnswer);
void SetVotedOnAnswer(unsigned short *pusVote, int nAnswer,
   unsigned short bVoted);
void ChooseAndVote(char bChangeAnswer);
void VoteOnAll(BOOL bForce);
BOOL VoteOnQuestion(int nQuestion, char bForceAnswer);
void ViewResults(void);
int GetQuestion(int nQuestion, tQuestionRecord *pQuestionRecord);
void AddQuestion(void);
void DeleteOrUndeleteQuestion(BOOL bDelete);
int NumQuestions(int nFromWhichQuestions);
int ChooseCategory(int nFromWhichQuestions, char *pszTitle, int *nLocation);
int ChooseQuestionInCategory(int nCategory, int nFromWhichQuestions,
   char *pszTitle, int *nLocation);
int ChooseQuestion(int nFromWhichQuestions, char *pszTitle, int *nLocation);
void DisplayQuestionResult(tQuestionRecord *pQuestionRecord);
int ReadOrAddCurrentUser(void);
void WriteCurrentUser(void);
FILE *ExclusiveFileOpen(char *pszFileName, char *pszMode, int *phHandle);
void ExclusiveFileClose(FILE *pfFile, int hHandle);
void WaitForEnter(void);
int UserFileHeaderChecksum(void);
void NoDropFileFunc(void);
BOOL ReadCategoryInfo(tCategoryInfo *pCategoryInfo);
void FreeCategoryArray(tCategoryInfo *pCategoryInfo);
tCategoryInfoRecord *AddCategoryRecord(tCategoryInfo *pCategoryInfo);
tCategoryInfoRecord *GetCategoryInfoRecord(tCategoryInfo *pCategoryInfo,
                                           char *pszCategory);
void SortCategoryInfoRecords(tCategoryInfo *pCategoryInfo);
int CompareCategoryRecords(const void *pElement1, const void *pElement2);
BOOL CompactDatabase();
void SysopCommands();


/* main() or WinMain() function - Program execution begins here. */
#ifdef ODPLAT_WIN32
int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
   LPSTR lpszCmdLine, int nCmdShow)
#else
int main(int nArgc, char *papszArgv[])
#endif
{
   /* Variable to store user's choice from the menu */
   char chMenuChoice;
   char bForceAnswer = FALSE;
   char bContinueInDoor = FALSE;
   char bInviteUser = FALSE;
   char *pszOption;
   int nArg;
   int nUnvotedQuestions;
   int nRow;
   int nPopupResult;
   int nCount;

   /* Enable use of OpenDoors configuration file system. */
   od_control.od_config_file = INCLUDE_CONFIG_FILE;

   /* Set function to process custom configuration file lines. */
   od_control.od_config_function = CustomConfigFunction;

   /* Set filename of configuration file. */
   od_control.od_config_filename = "EZVOTE.CFG";

   /* Include the OpenDoors multiple personality system, which allows    */
   /* the system operator to set the sysop statusline / function key set */
   /* to mimic the BBS software of their choice.                         */
   //od_control.od_mps = INCLUDE_MPS;
   
   /* Include the OpenDoors log file system, which will record when the */
   /* door runs, and major activites that the user performs.            */ 
   od_control.od_logfile = INCLUDE_LOGFILE;

   /* Set filename for log file. If not set, DOOR.LOG will be used by */
   /* default.                                                        */
   strcpy(od_control.od_logfile_name, "EZVOTE.LOG");
   
   /* Set program's name, to be written to the OpenDoors log file       */
   strcpy(od_control.od_prog_name, PROGRAM_NAME);
   strcpy(od_control.od_prog_version, PROGRAM_VERSION);
   strcpy(od_control.od_prog_copyright, PROGRAM_COPYRIGHT);

   /* Set function to be called before program exits. */
   od_control.od_before_exit = BeforeExitFunction;

#ifdef ODPLAT_WIN32
   /* In Windows, pass in nCmdShow value to OpenDoors. */
   od_control.od_cmd_show = nCmdShow;

   /* Mark unused parameters. */
   (void)hInstance;
   (void)hPrevInstance;

   od_control.od_cmd_line_help =
      "(Note that some options can be overriden by configuration or drop files.)\n"
      "\n"
      "-C or -CONFIG\t- Specfies configuration filename.\n"
      "-L or -LOCAL\t- Causes door to operate in local mode, without requiring a drop file.\n"
      "-DROPFILE x\t- Door information file directory and/or filename.\n"
      "-N x or -NODE x\t- Sets the node number to use.\n"
      "-INVITEUSER\t- Gives user the option of answering all new questions.\n"
      "-FORCEANSWER\t- Forces the user to answer all new questions.\n"
      "-CONTINUEINDOOR\t- Don't exit after -INVITEUSER or -FORCEANSWER; instead go to menu.\n"
      "-B x or -BPS x\t- Sets the serial port <---> modem bps (baud) rate to use.\n"
      "-P x or -PORT x\t- Sets the serial port to use, were 0=COM1, 1=COM2, etc.\n"
      "-HANDLE x\t- Provides an already open serial port handle.\n"
      "-MAXTIME x\t- Sets the maximum number of minutes that user will be permitted to access the door.\n"
      "-G or -GRAPHICS\t- Unless followed by 0 or N, turns on ANSI display mode.\n"
      "-BBSNAME x\t- Name of BBS.\n"
      "-USERNAME x\t- Name of user who is currently online.\n"
      "-TIMELEFT x\t- User's time remaining online.\n"
      "-SECURITY x\t- User's security level.\n"
      "-LOCATION x\t- Location from which user is calling.\n"
      "-?, -H or -HELP\t- Displays command-line help and exits.";

   od_parse_cmd_line(lpszCmdLine);
#else
   od_control.od_cmd_line_help =
      "(Some can be overriden by config/drop file)\n"
      " -C or -CONFIG    - Specfies configuration filename.\n"
      " -L or -LOCAL     - Causes door to operate in local mode, without requiring a\n"
      "                    door information (drop) file.\n"
      " -DROPFILE x      - Door information file directory and/or filename.\n"
      " -N x or -NODE x  - Sets the node number to use.\n"
      " -INVITEUSER      - Gives user the option of answering all new questions.\n"
      " -FORCEANSWER     - Forces the user to answer all new questions.\n"
      " -CONTINUEINDOOR  - No exit after -INVITEUSER or -FORCEANSWER; go to main menu.\n"
      " -B x or -BPS x   - Sets the serial port <---> modem bps (baud) rate to use.\n"
      " -P x or -PORT x  - Sets the serial port to use, were 0=COM1, 1=COM2, etc.\n"
      " -ADDRESS x       - Sets serial port address in HEXIDECIMAL (if no FOSSIL).\n"
      " -IRQ x           - Sets the serial port IRQ line (if FOSSIL is not used).\n"
      " -NOFOSSIL        - Disables use of FOSSIL driver, even if available.\n"
      " -NOFIFO          - Disables use of 16550 FIFO buffers (only if no FOSSIL).\n"
      " -MAXTIME x       - Sets the maximum number of minutes that any user will be\n"
      "                    permitted to access the door, regardless of time left.\n"
      " -G or -GRAPHICS  - Unless followed by 0 or N, turns on ANSI display mode.\n"
      " -BBSNAME x       - Name of BBS.\n"
      " -USERNAME x      - Name of user who is currently online.\n"
      " -TIMELEFT x      - User's time remaining online.\n"
      " -SECURITY x      - User's security level.\n"
      " -LOCATION x      - Location from which user is calling.\n"
      " -?, -H or -HELP  - Displays command-line help and exits.";

   od_parse_cmd_line(nArgc, papszArgv);
#endif

   /* Do not display OpenDoors copyright notice. */
   od_control.od_nocopyright = TRUE;

   /* Set fullscreen chat function. */
   od_control.od_cbefore_chat = fullscreen_chat;

   /* Set function to call if no drop file was found. */
   od_control.od_no_file_func = NoDropFileFunc;

   /* Initialize OpenDoors. This function call is optional, and can be used */
   /* to force OpenDoors to read the door informtion file and begin door    */
   /* operations. If a call to od_init() is not included in your program,   */
   /* OpenDoors initialization will be performed at the time of your first  */
   /* call to any OpenDoors function. */
   od_init();

   if(od_control.user_ansi || od_control.user_avatar)
   {
      od_control.od_after_chat = NULL;
   }

   /* Parse special mode command-line parameters. */
   for(nArg = 1; nArg < nArgc; ++nArg)
   {
      pszOption = papszArgv[nArg];
      
      if(*pszOption == '-' || *pszOption == '/')
      {
         ++pszOption;
      }
      
      if(stricmp(pszOption, "ForceAnswer") == 0)
      {
         bForceAnswer = TRUE;
      }
      else if(stricmp(pszOption, "InviteUser") == 0)
      {
         bInviteUser = TRUE;
      }
      else if(stricmp(pszOption, "ContinueInDoor") == 0)
      {
         bContinueInDoor = TRUE;
      }
   }

   /* Call the EZVote function ReadOrAddCurrentUser() to read the current */
   /* user's record from the EZVote user file, or to add the user to the  */
   /* file if this is the first time that they have used EZVote.          */
   if(!ReadOrAddCurrentUser())
   {
      /* If unable to obtain a user record for the current user, then exit */
      /* the door.                                                         */
      od_exit(1, FALSE);
   }
   
   /* If operating in "Force Answer" mode. */
   if(bForceAnswer || bAlwaysForceAnswer)
   {
      if(!bForceAnswer)
      {
         /* In this case, display the title screen now. */
         TitleScreen();
      }

      if(NumQuestions(QUESTIONS_NOT_VOTED_ON) > 0)
      {
         VoteOnAll(TRUE);
      }

      /* If operating in "Invite User" mode, then ask user whether or not */
      /* they want to continue into the door.                             */
      if(bInviteUser)
      {
         od_printf("`dark green`\n\rYou have now voted on all of the new EZVote online poll questions.\n\r\n\r");
         od_printf("`bright white`Would you like to enter the EZVote door now? (Y/N)");
         
         /* Continue in door if and only if user wants to. */
         bContinueInDoor = od_get_answer("YN") == 'Y';
      }
      else if(bAlwaysForceAnswer)
      {
         bContinueInDoor = TRUE;
      }

      /* Exit program right away, unless we were asked to do otherwise. */
      if(!bContinueInDoor)
      {
         od_exit(0, FALSE);
      }
   }
   
   /* If operating in "Invite User" mode without force answer mode. */
   else if(bInviteUser)
   {
      nUnvotedQuestions = NumQuestions(QUESTIONS_NOT_VOTED_ON);
   
      if(nUnvotedQuestions == 0)
      {
         od_printf("`dark green`\n\rThere are no new EZVote online poll questions that you haven't voted on.\n\r\n\r");
         od_exit(0, FALSE);
      }
      else
      {
         od_printf("`dark green`\n\rThere are %d EZVote online poll question(s) that you haven't voted on yet.\n\r\n\r",
            nUnvotedQuestions);
         od_printf("`bright white`Would you like to enter the EZVote door and vote on these questions now? (Y/N)");
      }

      /* Exit door if user doesn't want to vote. */
      if(od_get_answer("YN") == 'N')
      {
         od_printf("`white`\n\r\n\r");
         od_exit(0, FALSE);
      }

      /* Display the main title screen. */
      TitleScreen();
      
      /* If user does want to enter door, then begin them off in the vote */
      /* screen if there are any questions for them to vote on.          */
      if(nUnvotedQuestions > 0)
      {
         ChooseAndVote(FALSE);
      }
   }

   /* If we haven't already displayed the title screen, then do so now.*/
   TitleScreen();

   /* Generate main menu string if using new UI. */
   if(!bUseOldUI)
   {
      nCount = 1;

      strcpy(szMainMenu, "^Vote on a question");
      nVoteCmdIndex = nCount++;
      if(bAllowChange)
      {
         strcat(szMainMenu, "|^Change an answer");
         nChangeCmdIndex = nCount++;
      }

      if(bAllowAdd ||
         strcmp(od_control.sysop_name, od_control.user_name) == 0)
      {
         strcat(szMainMenu, "|^Add a new question");
         nAddCmdIndex = nCount++;
      }

      if(strcmp(od_control.sysop_name, od_control.user_name) == 0 ||
         bAllowUserDelete)
      {
         strcat(szMainMenu, "|^Delete a question");
         nDeleteCmdIndex = nCount++;
      }

      strcat(szMainMenu, "|View ^results of voting|^Page sysop for chat");
      nViewCmdIndex = nCount++;
      nPageCmdIndex = nCount++;

      if(strcmp(od_control.sysop_name, od_control.user_name) == 0)
      {
         strcat(szMainMenu, "|^Sysop Commands");
         nSysopCmdIndex = nCount++;
      }

      strcat(szMainMenu, "|^Quit to BBS|^Goodbye (end call)");
      nQuitCmdIndex = nCount++;
      nGoodbyeCmdIndex = nCount++;
   }

   /* Loop until the user choses to exit the door. For each iteration of  */
   /* this loop, we display the main menu, get the user's choice from the */
   /* menu, and perform the appropriate action for their choice.          */
   if(!bUseOldUI)
   {
      od_clr_scr();
      DisplayImbedded(NULL, szSplashANS, NULL, NULL);
   }

   for(;;)
   {
      if(bUseOldUI)
      {
         /* Clear the screen */
         od_clr_scr();

         /* Display main menu. */
   
         /* First, attempt to display menu from an EZVOTE.ASC/ANS/AVT/RIP file. */
         if((chMenuChoice = od_hotkey_menu("EZVOTE", "VRCADPQGS", TRUE)) == 0)
         {
            /* If there is not an EZVOTE.ASC/ANS/AVT/RIP file, then display */
            /* built-in main menu.                                          */
            if(od_control.user_ansi || od_control.user_avatar)
            {
               /* Display ANSI/AVATAR menu. */

               /* Display menu "background". */
               DisplayImbedded(NULL, szMainANS, NULL, NULL);

               /* Draw the appropriate menu options. */
               DrawMenuOption(22, 8, 'V', "Vote on a question");
               if(bAllowChange)
               {
                  DrawMenuOption(22, 11, 'C', "Change an answer");
               }

               nRow = 8;
               if(bAllowAdd ||
                  strcmp(od_control.sysop_name, od_control.user_name) == 0)
               {
                  DrawMenuOption(53, nRow, 'A', "Add a new question");
                  nRow += 3;
               }

               if(strcmp(od_control.sysop_name, od_control.user_name) == 0 ||
                  bAllowUserDelete)
               {
                  DrawMenuOption(53, nRow, 'D', "Delete a question");
                  nRow += 3;
               }

               if(nRow == 14 || bAllowChange)
               {
                  nRow = 15;
               }
               else
               {
                  nRow = 13;
               }

               DrawMenuOption(22, nRow, 'R', "View result of voting");
               DrawMenuOption(53, nRow, 'P', "Page sysop for chat");

               DrawMenuOption(22, 19, 'Q', "Quit to BBS");
               DrawMenuOption(53, 19, 'G', "Goodbye (end call)");

               /* Draw prompt. */
               od_set_cursor(23, 22);
               od_printf("`bright white`Press the key of your choice. (%d mins left)",
                  od_control.user_timelimit);
            }
            else
            {
               /* Display ASCII mode menu. */
               ClearWithHeader("Main Menu");

               od_printf("\n\r`dark green`");
               od_printf("                        [`bright green`V`dark green`] Vote on a question\n\r\n\r");

               if(bAllowChange)
               {
                  od_printf("                        [`bright green`C`dark green`] Change an answer\n\r\n\r");
               }

               od_printf("                        [`bright green`R`dark green`] View results of voting\n\r\n\r");

               /* Display Add New Question option if adding questions is enabled, */
               /* or if the system operator is using the door.                    */
               if(bAllowAdd ||
                  strcmp(od_control.sysop_name, od_control.user_name) == 0)
               {
                  od_printf("                        [`bright green`A`dark green`] Add a new question\n\r\n\r");
               }

               /* If current user is the system operator, add a D function to permit */
               /* deletion of unwanted questions.                                    */
               if(strcmp(od_control.sysop_name, od_control.user_name) == 0
                  || bAllowUserDelete)
               {
                  od_printf("                        [`bright green`D`dark green`] Delete a question\n\r\n\r");
               }
               od_printf("                        [`bright green`P`dark green`] Page system operator for chat\n\r\n\r");
               od_printf("                        [`bright green`Q`dark green`] Quit door and return to the BBS\n\r\n\r");
               od_printf("                        [`bright green`G`dark green`] Goodbye (end call)\n\r\n\r");
               od_printf("`bright white`Press the key of your choice. (%d mins left)\n\r`dark green`",
                  od_control.user_timelimit);
            }

            /* Get the user's choice from the main menu. This choice may only be */
            /* V, R, C, A, D, P, Q, G or S.                                      */
            chMenuChoice = od_get_answer("VRCADPQGS");
         }

         /* Perform the appropriate action based on the user's choice */
         switch(chMenuChoice)
         {
            case 'V':
               /* Call EZVote's function to vote on question */
               ChooseAndVote(FALSE);
               break;
         
            case 'R':
               /* Call EZVote's function to view the results of voting */
               ViewResults();
               break;
         
            case 'C':
               if(bAllowChange)
               {
                  ChooseAndVote(TRUE);
               }
               break;
         
            case 'A':
               /* Call EZVote's function to add a new question if door is */
               /* configured to allow the addition of question.           */
               if(bAllowAdd ||
                  strcmp(od_control.sysop_name, od_control.user_name) == 0)
               {
                  AddQuestion();
               }
               break;

            case 'D':
               /* Call EZVote's function to delete an existing question */
               DeleteOrUndeleteQuestion(TRUE);
               break;
         
            case 'P':
               /* If the user pressed P, allow them page the system operator. */
               od_page();
               break;

            case 'Q':
               CommandQuit();
               break;

            case 'G':
               CommandGoodbye();
               break;

            case 'S':
               if(strcmp(od_control.sysop_name, od_control.user_name) == 0)
               {
                  SysopCommands();
               }
               break;
         }
      }
      else
      {
         nPopupResult = od_popup_menu("Main Menu", szMainMenu, 34, 10,
            0, MENU_KEEP);

         if(nPopupResult == nVoteCmdIndex)
         {

         }
         else if(nPopupResult == nChangeCmdIndex)
         {

         }
         else if(nPopupResult == nAddCmdIndex)
         {

         }
         else if(nPopupResult == nDeleteCmdIndex)
         {

         }
         else if(nPopupResult == nViewCmdIndex)
         {

         }
         else if(nPopupResult == nPageCmdIndex)
         {

         }
         else if(nPopupResult == nQuitCmdIndex)
         {
            CommandQuit();
         }
         else if(nPopupResult == nGoodbyeCmdIndex)
         {
            CommandGoodbye();
         }
         else if(nPopupResult == nSysopCmdIndex)
         {
            if(strcmp(od_control.sysop_name, od_control.user_name) == 0)
            {
               SysopCommands();
            }
         }
      }
   }
}


void CommandGoodbye(void)
{
   /* If the user pressed G, ask whether they wish to hangup. */
   chYesOrNo = UserPrompt("End Call",
      "Are you sure you wish to hangup? (Y/N) ", NULL, "YN");

   /* If user answered yes, exit door and hangup */
   if(chYesOrNo == 'Y')
   {
      od_exit(0, TRUE);
   }
}


void CommandQuit(void)
{
   /* If we are to display the confirmation screen. */
   if(bConfirmDoorExit)
   {
      /* If the user pressed Q, confirm exit to BBS. */
      chYesOrNo = UserPrompt("Quit to BBS",
         "Do you wish to leave EZVote now? (Y/N) ", NULL, "YN");

      /* If the user pressed Y, exit door and return to BBS. */
      if(chYesOrNo == 'Y')
      {
         od_exit(0, FALSE);
      }
   }
   else
   {
      /* If confirmation screen is turned off, exit right away. */
      od_exit(0, FALSE);
   }
}


/* DrawMenuOption() - Draws an option on the ANSI/AVATAR built-in menu. */
void DrawMenuOption(int nCol, int nRow, char chCommandKey, char *pszOptionText)
{
   od_set_cursor(nRow, nCol);
   od_printf("`dark green`Ŀ");
   od_set_cursor(nRow + 1, nCol);
   od_printf(" `bright green`%c`dark green`  %s",
      chCommandKey, pszOptionText);
   od_set_cursor(nRow + 2, nCol);
   od_printf("");
}


/* TitleScreen() - Checks to see whether the title screen has been displayed */
/*                 and if not displays it now.                               */
void TitleScreen()
{
   char szLine[81];
   long lTimeInstalled;
   int nDaysInstalled;

   /* If title has already been displayed, then don't do anything. */
   if(bTitleDisplayed)
   {
      return;
   }

   /* If the user doesn't want a title screen, and they are registered, then */
   /* don't do anything.                                                     */
   if(!bShowTitleScreen && bUseOldUI)
   {
      return;
   }

   /* Ensure that the screen is cleared. */
   od_clr_scr();

   /* Display animated portion of the title screen. */
   od_control.od_emu_simulate_modem = TRUE;
   DisplayImbedded(szTitleASC, szTitleANS, NULL, NULL);
   od_control.od_emu_simulate_modem = FALSE;

   /* Display welcome line. */
   od_set_cursor(16, 1);
   od_printf(WELCOME_NAME);

   od_printf("\n\r\n\r                    Press [ENTER]/[RETURN] to enter EZVote.\n\r\n\r");

   /* Display copyright line. */
   od_printf(WELCOME_COPYRIGHT);

   /* Wait for the user to press enter or space. */
   od_get_answer("\n\r ");

   /* Record that the title screen has now been displayed. */
   bTitleDisplayed = TRUE;
}


void ClearWithHeader(char *pszHeaderText)
{
   char *pchHeaderChar;

   od_clr_scr();

   DisplayImbedded(szHeaderASC, szHeaderANS, NULL, NULL);

   od_repeat(' ', (80 - 27) - (strlen(pszHeaderText) * 2));

   for(pchHeaderChar = pszHeaderText; *pchHeaderChar; ++pchHeaderChar)
   {
      od_putch(toupper(*pchHeaderChar));
      od_putch(' ');
   }
   od_printf("`dark green`\n\r");
}


char UserPrompt(char *pszHeaderText, char *pszMessageLine1,
   char *pszMessageLine2, char *pszPossibleKeys)
{
   char chToReturn;
   void *pWindow;
   int nWindowWidth;
   int nWindowLeft;
   int nWindowTop;
   int nWindowHeight;

   if(pszMessageLine2 == NULL)
   {
      pszMessageLine2 = "";
   }

   if(bUseOldUI)
   {
      MessageScreen(pszHeaderText, pszMessageLine1, pszMessageLine2);
   }
   else
   {
      nWindowWidth = MAX(strlen(pszMessageLine1), strlen(pszMessageLine2))
         + 4;
      nWindowLeft = (80 - nWindowWidth) / 2;

      if(strlen(pszMessageLine2) > 0)
      {
         nWindowHeight = 6;
      }
      else
      {
         nWindowHeight = 5;
      }
      nWindowTop = (24 - nWindowHeight) / 2;

      pWindow = od_window_create(nWindowLeft, nWindowTop,
         nWindowLeft + nWindowWidth - 1,
         nWindowTop + nWindowHeight - 1, pszHeaderText,
         btWinBrdrColor, btWinTitleColor, btWinTextColor, 0);

      od_set_attrib(btWinTextColor);

      od_set_cursor(nWindowTop + 2, (80 - strlen(pszMessageLine1)) / 2);
      od_disp_str(pszMessageLine1);

      if(strlen(pszMessageLine2) > 0)
      {
         od_set_cursor(nWindowTop + 3, (80 - strlen(pszMessageLine2)) / 2);
         od_disp_str(pszMessageLine2);
      }
   }

   chToReturn = od_get_answer(pszPossibleKeys);

   if(!bUseOldUI)
   {
      od_window_remove(pWindow);
   }

   return(chToReturn);
}


void MessageScreen(char *pszHeaderText, char *pszMessageLine1,
   char *pszMessageLine2)
{
   ClearWithHeader(pszHeaderText);
   od_repeat('\n', 7);
   od_repeat(' ', (80 - strlen(pszMessageLine1)) / 2);
   od_disp_str(pszMessageLine1);
   if(pszMessageLine2 != NULL && strlen(pszMessageLine2) > 0)
   {
      od_printf("\n\r");
      od_repeat(' ', (80 - strlen(pszMessageLine2)) / 2);
      od_disp_str(pszMessageLine2);
   }
}


/* CustomConfigFunction() is called by OpenDoors to process custom */
/* configuration file keywords that EZVote uses.                   */
void CustomConfigFunction(char *pszKeyword, char *pszOptions)
{
   if(stricmp(pszKeyword, "ViewUnanswered") == 0)
   {
      /* If keyword is ViewUnanswered, set local variable based on contents */
      /* of options string.                                                 */
      if(stricmp(pszOptions, "Yes") == 0)
      {
         nViewResultsFrom = QUESTIONS_VOTED_ON | QUESTIONS_NOT_VOTED_ON;
      }
      else if(stricmp(pszOptions, "No") == 0)
      {
         nViewResultsFrom = QUESTIONS_VOTED_ON;
      }
   }
   else if(stricmp(pszKeyword, "AllowAdd") == 0)
   {
      /* If keyword is ViewUnvoted, set local variable based on contents */
      /* of options string.                                              */
      if(stricmp(pszOptions, "Yes") == 0)
      {
         bAllowAdd = TRUE;
      }
      else if(stricmp(pszOptions, "No") == 0)
      {
         bAllowAdd = FALSE;
      }
   }
   else if(stricmp(pszKeyword, "AllowChange") == 0)
   {
      /* If keyword is AllowChange, set local variable based on contents */
      /* of options string.                                              */
      if(stricmp(pszOptions, "Yes") == 0)
      {
         bAllowChange = TRUE;
      }
      else if(stricmp(pszOptions, "No") == 0)
      {
         bAllowChange = FALSE;
      }
   }
   else if(stricmp(pszKeyword, "AllowUserDelete") == 0)
   {
      /* If keyword is AllowUserDelete, set local variable based on */
      /* contents of options string.                                */
      if(stricmp(pszOptions, "Yes") == 0)
      {
         bAllowUserDelete = TRUE;
      }
      else if(stricmp(pszOptions, "No") == 0)
      {
         bAllowUserDelete = FALSE;
      }
   }
   else if(stricmp(pszKeyword, "AlwaysForceAnswer") == 0)
   {
      /* If keyword is AlwaysForceAnswer, set local variable based on */
      /* contents of options string.                                  */
      if(stricmp(pszOptions, "Yes") == 0)
      {
         bAlwaysForceAnswer = TRUE;
      }
      else if(stricmp(pszOptions, "No") == 0)
      {
         bAlwaysForceAnswer = FALSE;
      }
   }
   else if(stricmp(pszKeyword, "ShowTitleScreen") == 0)
   {
      if(stricmp(pszOptions, "Yes") == 0)
      {
         bShowTitleScreen = TRUE;
      }
      else if(stricmp(pszOptions, "No") == 0)
      {
         bShowTitleScreen = FALSE;
      }
   }
   else if(stricmp(pszKeyword, "ConfirmDoorExit") == 0)
   {
      if(stricmp(pszOptions, "Yes") == 0)
      {
         bConfirmDoorExit = TRUE;
      }
      else if(stricmp(pszOptions, "No") == 0)
      {
         bConfirmDoorExit = FALSE;
      }
   }
   else if(stricmp(pszKeyword, "ShowSingleCategory") == 0)
   {
      if(stricmp(pszOptions, "Yes") == 0)
      {
         bShowSingleCategory = TRUE;
      }
      else if(stricmp(pszOptions, "No") == 0)
      {
         bShowSingleCategory = FALSE;
      }
   }
   else if(stricmp(pszKeyword, "AllowUserCategoryCreation") == 0)
   {
      if(stricmp(pszOptions, "Yes") == 0)
      {
         bAllowUserCategoryCreation = TRUE;
      }
      else if(stricmp(pszOptions, "No") == 0)
      {
         bAllowUserCategoryCreation = FALSE;
      }
   }
   else if(stricmp(pszKeyword, "UseCategories") == 0)
   {
      if(stricmp(pszOptions, "Yes") == 0)
      {
         bUseCategories = TRUE;
      }
      else if(stricmp(pszOptions, "No") == 0)
      {
         bUseCategories = FALSE;
      }
   }
}


/* EZVote configures OpenDoors to call the BeforeExitFunction() before    */
/* the door exists for any reason. You can use this function to close any */
/* files or perform any other operations that you wish to have peformed   */
/* before OpenDoors exists for any reason. The od_control.od_before_exit  */
/* variable sets the function to be called before program exit.           */
void BeforeExitFunction(void)
{
   char szLogMessage[80];
   
   /* Write number of messages voted on to log file. */
   sprintf(szLogMessage, "User has voted on %d question(s)",
      nQuestionsVotedOn);
   od_log_write(szLogMessage);
}


unsigned short HasVotedOnAnswer(unsigned short usVote, int nAnswer)
{
   unsigned short usAnswerMask = 0x0001 << nAnswer;

   return(usVote & usAnswerMask);
}


void SetVotedOnAnswer(unsigned short *pusVote, int nAnswer, unsigned short bVoted)
{
   unsigned short usAnswerMask = 0x0001 << nAnswer;

   if(bVoted)
   {
      *pusVote |= usAnswerMask;
   }
   else
   {
      *pusVote &= ~usAnswerMask;
   }
}


/* EZVote calls the ChooseAndVote() function when the user chooses the    */
/* vote command from the main menu. This function displays a list of      */
/* available topics, asks for the user's answer to the topic they select, */
/* and display's the results of voting on that topic.                     */
void ChooseAndVote(char bChangeAnswer)
{
   int nQuestion;
   tQuestionPos QuestionPos;

   ResetQuestionPos(&QuestionPos);

   /* Loop until the user chooses to return to the main menu, or until */
   /* there are no more questions to vote on.                          */
   for(;;)   
   {
      /* Allow the user to choose a question from the list of questions */
      /* that they have not voted on.                                   */
      nQuestion = ChooseQuestion(bChangeAnswer ? QUESTIONS_VOTED_ON :
         (QUESTIONS_NOT_VOTED_ON | VOTE_ON_ALL), bChangeAnswer ?
         "Change An Answer" :
         "Vote On A Question",
         &QuestionPos);
   

      /* If the user did not choose a question, return to main menu. */   
      if(nQuestion == NO_QUESTION)
      {
         return;
      }
      else if(nQuestion == ALL_QUESTIONS)
      {
         VoteOnAll(FALSE);
         continue;
      }
      
      /* Now, let the user vote on this quesiton. */
      VoteOnQuestion(nQuestion, FALSE);
   }
}


void VoteOnAll(BOOL bForce)
{
   tQuestionRecord QuestionRecord;
   FILE *fpQuestionFile;
   int hQuestionFile;
   char bVotedOnQuestion;
   int nQuestion = 0;
   int nTotalQuestions;
   char abVoteOn[MAX_QUESTIONS];
   
   /* Attempt to open question file. */
   fpQuestionFile = ExclusiveFileOpen(QUESTION_FILENAME, "r+b", &hQuestionFile);

   /* If unable to open question file, assume that no questions have been */
   /* created.                                                            */
   if(fpQuestionFile == NULL)
   {
      return;
   }
   
   /* Loop for every question record in the file, determining which */
   /* questions the user will have to vote on.                      */
   while(fread(&QuestionRecord, sizeof(QuestionRecord), 1, fpQuestionFile) == 1)
   {
      /* Determine whether or not the user has voted on this question. */
      bVotedOnQuestion = (CurrentUserRecord.ausVotes[nQuestion] != 0);
   
      /* The user will be required to vote on this question if they */
      /* haven't voted on it already, and it is not deleted.        */
      abVoteOn[nQuestion] = !bVotedOnQuestion && !QuestionRecord.bDeleted;
   
      ++nQuestion;
   }
   
   ExclusiveFileClose(fpQuestionFile, hQuestionFile);
   
   nTotalQuestions = nQuestion;
   
   /* Loop for each question record. */
   for(nQuestion = 0; nQuestion < nTotalQuestions; ++nQuestion)
   {
      /* If the user has to vote on this question. */
      if(abVoteOn[nQuestion])
      {
         /* Enter vote mode, possibly in "force answer" mode. */
         if(!VoteOnQuestion(nQuestion, bForce))
         {
            return;
         }
      }
   }
}


BOOL VoteOnQuestion(int nQuestion, char bForceAnswer)
{
   int nAnswer;
   tQuestionRecord QuestionRecord;
   char aszNewAnswer[MAX_ANSWERS][ANSWER_STR_SIZE];
   int nNewAnswers = 0;
   char szUserInput[3];
   FILE *fpFile;
   int hFile;
   unsigned short usOriginalAnswer;

   /* Read the question chosen by the user. */
   if(!GetQuestion(nQuestion, &QuestionRecord))
   {
      /* If unable to access file, return to main menu. */
      return(FALSE);
   }

   /* Don't allow addition of new answers if maximum number of answers */
   /* have already been added.                                         */
   if(QuestionRecord.nTotalAnswers >= MAX_ANSWERS)
   {
      QuestionRecord.bCanAddAnswers = FALSE;
   }
   
   /* Store user's original response for this question. */
   usOriginalAnswer = CurrentUserRecord.ausVotes[nQuestion];

   /* Loop until user makes a valid respose. */
   for(;;)
   {
      /* Display question to user. */

      /* Clear the screen. */
      od_clr_scr();

      /* Display question itself. */
      od_printf("`bright red`%s\n\r\n\r", QuestionRecord.szQuestion);

      /* Loop for each answer to the question. */   
      for(nAnswer = 0; nAnswer < QuestionRecord.nTotalAnswers; ++nAnswer)
      {
         /* Display answer number and answer. */
         od_printf("`bright green`%d. `dark green`%s",
            nAnswer + 1,
            QuestionRecord.aszAnswer[nAnswer]);
         if(HasVotedOnAnswer(CurrentUserRecord.ausVotes[nQuestion],
            nAnswer))
         {
            od_printf("`bright red`  [Voted For]");
         }
         od_printf("\n\r");
      }

      /* Display prompt to user. */
      if(QuestionRecord.bMultipleAnswers)
      {
         od_printf("\n\r`bright white`You can vote for more than one answer for this question.\n\r\n\r");
         od_printf("`bright white`Enter answer number to vote/unvote on");
      }
      else
      {
         od_printf("\n\r`bright white`Enter answer number");
      }
      
      if(QuestionRecord.bCanAddAnswers)
      {
         od_printf(", [A] to Add your own response");
      }
      
      if(QuestionRecord.bMultipleAnswers)
      {
         if(bForceAnswer)
         {
            od_printf(",\n\r[F] to Finalize your vote: `dark green`");
         }
         else
         {
            od_printf(",\n\r[C] to Cancel, [F] to Finalize your vote: `dark green`");
         }
      }
      else
      {
         if(bForceAnswer)
         {
            od_printf(": `dark green`");
         }
         else
         {
            od_printf(", [C] to Cancel: `dark green`");
         }
      }

      /* Get response from user. */
      od_input_str(szUserInput, 2, ' ', 255);
      /* Add a blank line. */      
      od_printf("\n\r");

      /* If user entered Q, return to main menu. */
      if(stricmp(szUserInput, "C") == 0 && !bForceAnswer)
      {
         CurrentUserRecord.ausVotes[nQuestion] = usOriginalAnswer;
         return(FALSE);
      }
      
      if(QuestionRecord.bMultipleAnswers && stricmp(szUserInput, "F") == 0)
      {
         /* If the user has not voted for any answers. */
         if(!(CurrentUserRecord.ausVotes[nQuestion] & 0x7fff))
         {
            /* Display message. */
            od_printf("`bright white`You have not chosen any answers to vote on. Are you sure that you wish to\n\r");
            od_printf("pass on this question? (Y/N) `dark green`");
            if(od_get_answer("YN") == 'N')
            {
               continue;
            }

            /* High-order bit indicates pass on question. */
            CurrentUserRecord.ausVotes[nQuestion] = 0x8000;
         }
         break;
      }

      /* If user enetered A, and adding answers is premitted ... */
      else if (stricmp(szUserInput, "A") == 0
         && QuestionRecord.bCanAddAnswers)
      {
         /* ... Prompt for answer from user. */
         od_printf("`bright green`Please enter your new answer:\n\r");
         od_printf("`dark green`[------------------------------]\n\r ");
      
         /* Get string from user. */
         od_input_str(aszNewAnswer[nNewAnswers], ANSWER_STR_SIZE - 1, ' ', 255);

         /* If user entered an invalid answer, then restart loop. */
         if(strlen(aszNewAnswer[nNewAnswers]) == 0)
         {
            continue;
         }
         
         /* Add answer to current complete list of answers for this */
         /* question.                                               */ 
         nAnswer = QuestionRecord.nTotalAnswers++;
         strcpy(QuestionRecord.aszAnswer[nAnswer], aszNewAnswer[nNewAnswers]);

         /* Increment count of new answers added by the user. */            
         nNewAnswers++;
      }
      else
      {
         /* Otherwise, attempt to get answer number from user. */      
         nAnswer = atoi(szUserInput) - 1;
      }

      /* If user input is not a valid answer. */      
      if(nAnswer < 0 || nAnswer >= QuestionRecord.nTotalAnswers)
      {
         /* Display message. */
         od_printf("That is not a valid response.\n\r");
         WaitForEnter();
         
         /* Display possible answers again. */
         continue;
      }
      
      if(QuestionRecord.bMultipleAnswers)
      {
         /* Toggle the state of this answer. */
         SetVotedOnAnswer(&CurrentUserRecord.ausVotes[nQuestion], nAnswer,
            !HasVotedOnAnswer(CurrentUserRecord.ausVotes[nQuestion],
            nAnswer));
      }
      else
      {
         /* Set state of all answers to FALSE. */
         CurrentUserRecord.ausVotes[nQuestion] = 0x0000;
         
         /* Set state of this answer to TRUE. */
         SetVotedOnAnswer(&CurrentUserRecord.ausVotes[nQuestion],
            nAnswer, TRUE);

         /* Exit loop, rather than asking for answer. */
         break;
      }
   }

   /* Update question with user's vote. */

   /* Open question file for exclusive access by this node. */
   fpFile = ExclusiveFileOpen(QUESTION_FILENAME, "r+b", &hFile);
   if(fpFile == NULL)
   {
      /* If unable to access file, display error and return. */
      od_log_write("ERROR 001: Disk error or question file locked");
      MessageScreen("Error",
         "Unable to access the question file to record your vote.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      CurrentUserRecord.ausVotes[nQuestion] = usOriginalAnswer;
      return(FALSE);
   }

   /* Read the answer record from disk, because it may have been changed */
   /* by another node.                                                   */
   fseek(fpFile, (long)nQuestion * sizeof(tQuestionRecord), SEEK_SET);
   if(fread(&QuestionRecord, sizeof(tQuestionRecord), 1, fpFile) != 1)
   {
      /* If unable to access file, display error and return. */
      ExclusiveFileClose(fpFile, hFile);

      od_log_write("ERROR 002: Unable to read from question file");
      MessageScreen("Error",
         "Unable to access the question file to record your vote.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");

      CurrentUserRecord.ausVotes[nQuestion] = usOriginalAnswer;
      return(FALSE);
   }

   /* If user entered new answer(s), try to add them to the question. */
   if(nNewAnswers > 0)
   {
      for(nAnswer = 0; nAnswer < nNewAnswers; ++nAnswer)
      {
         /* Check that there is still room for another answer. */
         if(QuestionRecord.nTotalAnswers >= MAX_ANSWERS)
         {
            ExclusiveFileClose(fpFile, hFile);
            MessageScreen("Error",
               "Sorry, this question already has the maximum number of answers.",
               "Press [Enter] to continue.");
            od_get_answer("\n\r");
            CurrentUserRecord.ausVotes[nQuestion] = usOriginalAnswer;
            return(FALSE);
         }
   
         /* Initialize new answer string and count. */
         strcpy(QuestionRecord.aszAnswer[QuestionRecord.nTotalAnswers],
            aszNewAnswer[nAnswer]);
         QuestionRecord.auVotesForAnswer[QuestionRecord.nTotalAnswers] = 0;
   
         /* Add 1 to total number of answers. */
         ++QuestionRecord.nTotalAnswers;
      }
   }
   
   /* Remove any of the user's previous votes on question. */
   for(nAnswer = 0; nAnswer < QuestionRecord.nTotalAnswers; ++nAnswer)
   {
      if(HasVotedOnAnswer(usOriginalAnswer, nAnswer))
      {
         --QuestionRecord.auVotesForAnswer[nAnswer];
      }
   }

   /* Add user's vote to question. */
   for(nAnswer = 0; nAnswer < QuestionRecord.nTotalAnswers; ++nAnswer)
   {
      if(HasVotedOnAnswer(CurrentUserRecord.ausVotes[nQuestion], nAnswer))
      {
         ++QuestionRecord.auVotesForAnswer[nAnswer];
      }
   }
   
   /* If user had not originally voted on this question, then add one */
   /* to number of users who have voted on this question.             */
   if(!usOriginalAnswer)
   {
      ++QuestionRecord.uTotalVotes;
   }

   /* Write the question record back to the file. */
   fseek(fpFile, (long)nQuestion * sizeof(tQuestionRecord), SEEK_SET);
   if(fwrite(&QuestionRecord, sizeof(tQuestionRecord), 1, fpFile) != 1)
   {
      /* If unable to access file, display error and return. */
      ExclusiveFileClose(fpFile, hFile);
      od_log_write("ERROR 003: Unable to write to question file");
      MessageScreen("Error",
         "Unable to access the question file to record your vote.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      CurrentUserRecord.ausVotes[nQuestion] = usOriginalAnswer;
      return(FALSE);
   }

   /* Close the question file to allow access by other nodes. */
   ExclusiveFileClose(fpFile, hFile);

   /* Open user file for exclusive access by this node. */
   fpFile = ExclusiveFileOpen(USER_FILENAME, "r+b", &hFile);
   if(fpFile == NULL)
   {
      /* If unable to access file, display error and return. */
      od_log_write("ERROR 004: Disk error or user file is locked");
      MessageScreen("Error",
         "Unable to access the user file to record your vote.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      CurrentUserRecord.ausVotes[nQuestion] = usOriginalAnswer;
      return(FALSE);
   }

   /* Update the user's record in the user file. */
   fseek(fpFile, sizeof(tUserFileHeader) +
      ((long)nCurrentUserNumber * sizeof(tUserRecord)), SEEK_SET);
   if(fwrite(&CurrentUserRecord, sizeof(tUserRecord), 1, fpFile) != 1)
   {
      /* If unable to access file, display error and return. */
      ExclusiveFileClose(fpFile, hFile);
      od_log_write("ERROR 005: Unabe to write to user file");
      MessageScreen("Error",
         "Unable to access the user file to record your vote.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      return(FALSE);
   }

   /* Close the user file to allow access by other nodes. */
   ExclusiveFileClose(fpFile, hFile);

   if(!bForceAnswer)
   {
      /* Display the result of voting on this question to the user. */
      DisplayQuestionResult(&QuestionRecord);
   }

   /* Add 1 to count of questions that the user has voted on. */
   nQuestionsVotedOn++;

   return(TRUE);
}


/* The ViewResults function is called when the user chooses the "view    */
/* results" command from the main menu. This function alows the user to  */
/* choose a question from the list of questions, and then displays the   */
/* results of voting on that question.                                   */
void ViewResults(void)
{
   int nChoice;
   tQuestionRecord QuestionRecord;
   tQuestionPos QuestionPos;

   ResetQuestionPos(&QuestionPos);

   /* Loop until user chooses to return to main menu. */
   for(;;)
   {   
      /* Allow the user to choose a question from the list of questions that */
      /* they have already voted on.                                         */
      nChoice = ChooseQuestion(nViewResultsFrom,
         "View Results", &QuestionPos);

      /* If the user did not choose a question, return to main menu. */   
      if(nChoice == NO_QUESTION)
      {
         return;
      }
   
      /* Read the specified question number from the question file. */
      if(!GetQuestion(nChoice, &QuestionRecord))
      {
         return;
      }
   
      /* Display the results for the selected question. */
      DisplayQuestionResult(&QuestionRecord);
   }
}


/* The GetQuestion function read the record for the specified question */
/* number from the question file.                                      */
int GetQuestion(int nQuestion, tQuestionRecord *pQuestionRecord)
{
   FILE *fpQuestionFile;
   int hQuestionFile;

   /* Open the question file for exculsive access by this node. */
   fpQuestionFile = ExclusiveFileOpen(QUESTION_FILENAME, "r+b", &hQuestionFile);
   if(fpQuestionFile == NULL)
   {
      /* If unable to access file, display error and return. */
      od_log_write("ERROR 006: Disk error or question file is locked");
      MessageScreen("Error",
         "Unable to access the question file.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      return(FALSE);
   }
   
   /* Move to location of question in file. */
   fseek(fpQuestionFile, (long)nQuestion * sizeof(tQuestionRecord), SEEK_SET);
   
   /* Read the question from the file. */
   if(fread(pQuestionRecord, sizeof(tQuestionRecord), 1, fpQuestionFile) != 1)
   {
      /* If unable to access file, display error and return. */
      ExclusiveFileClose(fpQuestionFile, hQuestionFile);
      od_log_write("ERROR 007: Unable to read from question file");
      MessageScreen("Error",
         "Unable to access the question file.",
         "Press [Enter] to continue.");
      return(FALSE);;
   }
   
   /* Close the question file to allow access by other nodes. */
   ExclusiveFileClose(fpQuestionFile, hQuestionFile);

   /* Return with success. */
   return(TRUE);
}
 

/* The AddQuestion() function is called when the user chooses the "add    */
/* question" option from the main menu. This function allows the user     */
/* to enter a new question, possible responses, and save the question for */
/* other users to vote on.                                                */
void AddQuestion(void)
{
   tQuestionRecord QuestionRecord;
   FILE *fpQuestionFile;
   int hQuestionFile;
   char szLogMessage[100];
   int nQuestions;
   int nMaxQuestions;

   /* Display screen header. */
   ClearWithHeader("Add A Question");
   od_printf("\n\r");

   /* Set category of new question to general. */
   strcpy(QuestionRecord.szCategory, "General");

   /* Reset unused field. */
   QuestionRecord.lUnused = 0;
   
   /* Obtain quesiton text from the user. */
   od_printf("`bright green`Enter Your Question (blank line cancels)\n\r");
   od_printf("`dark green`[----------------------------------------------------------------------]\n\r ");
   od_input_str(QuestionRecord.szQuestion, QUESTION_STR_SIZE - 1, ' ', 255);
   
   /* If question was empty, then return to main menu. */
   if(strlen(QuestionRecord.szQuestion) == 0)
   {
      return;
   }
   
   /* Display prompt for answers. */
   od_printf("\n\r`bright green`Enter Possible Answers (blank line when done)\n\r");
   od_printf("`dark green`   [------------------------------]\n\r");
   
   /* Loop, getting answers from user. */
   for(QuestionRecord.nTotalAnswers = 0;
       QuestionRecord.nTotalAnswers < MAX_ANSWERS;
       QuestionRecord.nTotalAnswers++)
   {
      /* Display prompt with answer number. */
      od_printf("`bright green`%2d: `dark green`", QuestionRecord.nTotalAnswers + 1);
      
      /* Get string from user. */
      od_input_str(QuestionRecord.aszAnswer[QuestionRecord.nTotalAnswers],
         ANSWER_STR_SIZE - 1, ' ', 255);
         
      /* If string was empty, then exit loop. */
      if(strlen(QuestionRecord.aszAnswer[QuestionRecord.nTotalAnswers]) == 0)
      {
         break;
      }
      
      /* Reset count of votes for this answer to zero. */
      QuestionRecord.auVotesForAnswer[QuestionRecord.nTotalAnswers] = 0;
   }
   
   /* If no answers were supplied, then cancel, returning to main menu. */
   if(QuestionRecord.nTotalAnswers == 0)
   {
      return;
   }

   /* Ask whether users should be able to add their own answers. */
   od_printf("\n\r`bright green`Should voters be permitted to add their own options? (Y/N) `dark green`");
   
   /* Get answer from user. */
   if(od_get_answer("YN") == 'Y')
   {
      /* If user pressed the 'Y' key. */
      od_printf("Yes\n\r\n\r");
      
      /* Record user's response. */
      QuestionRecord.bCanAddAnswers = TRUE;
   }
   else
   {
      /* If user pressed the 'N' key. */
      od_printf("No\n\r\n\r");

      /* Record user's response. */
      QuestionRecord.bCanAddAnswers = FALSE;
   }
   
   /* Ask whether users should be able to add their own answers. */
   od_printf("`bright green`Should a voter be permitted to vote for more than one answer? (Y/N) `dark green`");
   
   /* Get answer from user. */
   if(od_get_answer("YN") == 'Y')
   {
      /* If user pressed the 'Y' key. */
      od_printf("Yes\n\r\n\r");
      
      /* Record user's response. */
      QuestionRecord.bMultipleAnswers = TRUE;
   }
   else
   {
      /* If user pressed the 'N' key. */
      od_printf("No\n\r\n\r");

      /* Record user's response. */
      QuestionRecord.bMultipleAnswers = FALSE;
   }
   
   /* Ask whether users should be able to add their own answers. */
   od_printf("`bright green`As the creator of this question, do you wish to remain anonymous? (Y/N) `dark green`");
   
   /* Get answer from user. */
   if(od_get_answer("YN") == 'Y')
   {
      /* If user pressed the 'Y' key. */
      od_printf("Yes\n\r\n\r");
      
      /* Record user's response. */
      QuestionRecord.bAnonymous = TRUE;
   }
   else
   {
      /* If user pressed the 'N' key. */
      od_printf("No\n\r\n\r");

      /* Record user's response. */
      QuestionRecord.bAnonymous = FALSE;
   }

   /* Confirm save of new question. */
   od_printf("`bright green`Do you wish to [S]ave or [D]iscard this new question? (S/D) `dark green`");
   
   /* If user does not want to save the question, return to main menu now. */
   if(od_get_answer("SD") == 'D')
   {
      return;
   }

   /* Set total number of votes for this question to 0. */   
   QuestionRecord.uTotalVotes = 0;
   
   /* Set creator name and creation time for this question. */
   strcpy(QuestionRecord.szCreatorName, od_control.user_name);
   QuestionRecord.lCreationTime = time(NULL);
   QuestionRecord.bDeleted = FALSE;
   
   /* Open question file for exclusive access by this node. */
   fpQuestionFile = ExclusiveFileOpen(QUESTION_FILENAME, "a+b", &hQuestionFile);
   if(fpQuestionFile == NULL)
   {
      od_log_write("ERROR 008: Disk error or question file is locked");
      MessageScreen("Error",
         "Unable to access the question file to add your question.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      return;
   }
   
   /* Determine number of records in question file. */
   fseek(fpQuestionFile, 0, SEEK_END);
   
   /* If question file is full, display message and return to main menu */
   /* after closing file.                                               */
   nQuestions = ftell(fpQuestionFile) / sizeof(tQuestionRecord);
   nMaxQuestions = MAX_QUESTIONS;
   if(nQuestions >= nMaxQuestions)
   {
      ExclusiveFileClose(fpQuestionFile, hQuestionFile);
      od_log_write("Unable to add another question; question file is full");
      MessageScreen("Add A Question",
         "Cannot add another question, question file is full.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      return;
   }
   
   /* Add new question to file. */
   if(fwrite(&QuestionRecord, sizeof(QuestionRecord), 1, fpQuestionFile) != 1)
   {
      ExclusiveFileClose(fpQuestionFile, hQuestionFile);
      od_log_write("ERROR 009: Unable to write to question file");
      MessageScreen("Error",
         "Unable to access the question file to add your question.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      return;
   }
   
   /* Close question file, allowing other nodes to access file. */
   ExclusiveFileClose(fpQuestionFile, hQuestionFile);

   /* Record in the logfile that user has added a new question. */
   sprintf(szLogMessage, "User adding questions: %s",
      QuestionRecord.szQuestion);
   od_log_write(szLogMessage);
}


void DeleteOrUndeleteQuestion(BOOL bDelete)
{
   int nQuestion;
   tQuestionRecord QuestionRecord;
   FILE *fpFile;
   int hFile;
   int nChooseQuestionsFrom;
   tQuestionPos QuestionPos;

   ResetQuestionPos(&QuestionPos);

   /* Check that user is system operator. */
   if(strcmp(od_control.user_name, od_control.sysop_name) != 0
      && !bAllowUserDelete)
   {
      return;
   }

   if(bDelete)
   {   
      if(strcmp(od_control.user_name, od_control.sysop_name) == 0)
      {
         nChooseQuestionsFrom = QUESTIONS_NOT_VOTED_ON | QUESTIONS_VOTED_ON;
      }
      else
      {
         nChooseQuestionsFrom = QUESTIONS_AUTHORED;
      }
   }
   else
   {
      nChooseQuestionsFrom = QUESTIONS_DELETED;
   }

   /* Allow the user to choose a question from the list of all questions. */
   nQuestion = ChooseQuestion(nChooseQuestionsFrom,
      bDelete ? "Delete A Question" : "Undelete A Question", &QuestionPos);

   /* If the user did not choose a question, return to main menu. */   
   if(nQuestion == NO_QUESTION)
   {
      return;
   }

   /* Read the question chosen by the user. */
   if(!GetQuestion(nQuestion, &QuestionRecord))
   {
      /* If unable to access file, return to main menu. */
      return;
   }

   /* Confirm deletion of this question, if operating in delete mode. */
   if(bDelete)
   {
      ClearWithHeader("Delete A Question");
      od_repeat('\n', 7);
      
      od_printf("\n\r`bright green`Are you sure you want to delete the question:\n\r   `dark green`%s\n\r",
         QuestionRecord.szQuestion);
      od_printf("`bright green`[Y]es or [N]o?\n\r`dark green`");

      /* If user canceled deletion, return now. */   
      if(od_get_answer("YN") == 'N')
      {
         return;
      }
   }

   /* Mark the question as being deleted or undeleted. */
   QuestionRecord.bDeleted = bDelete;
   
   /* Open question file for exclusive access by this node. */
   fpFile = ExclusiveFileOpen(QUESTION_FILENAME, "r+b", &hFile);
   if(fpFile == NULL)
   {
      /* If unable to access file, display error and return. */
      od_log_write("ERROR 010: Disk error or question file is locked");
      MessageScreen("Error",
         "Unable to access the question file.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      return;
   }

   /* Write the question record back to the file. */
   fseek(fpFile, (long)nQuestion * sizeof(tQuestionRecord), SEEK_SET);
   if(fwrite(&QuestionRecord, sizeof(tQuestionRecord), 1, fpFile) != 1)
   {
      /* If unable to access file, display error and return. */
      ExclusiveFileClose(fpFile, hFile);
      od_log_write("ERROR 011: Unable to write to question file");
      MessageScreen("Error",
         "Unable to access the question file.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      return;
   }
   
   /* Close the question file to allow access by other nodes. */
   ExclusiveFileClose(fpFile, hFile);
}


int NumQuestions(int nFromWhichQuestions)
{
   int nQuestions = 0;
   tQuestionRecord QuestionRecord;
   FILE *fpQuestionFile;
   int hQuestionFile;
   char bVotedOnQuestion;
   int nFileQuestion = 0;
   
   /* Attempt to open question file. */
   fpQuestionFile = ExclusiveFileOpen(QUESTION_FILENAME, "r+b", &hQuestionFile);

   /* If unable to open question file, assume that no questions have been */
   /* created.                                                            */
   if(fpQuestionFile == NULL)
   {
      return(0);
   }
   
   /* Loop for every question record in the file. */
   while(fread(&QuestionRecord, sizeof(QuestionRecord), 1, fpQuestionFile) == 1)
   {
      /* Determine whether or not the user has voted on this question. */
      bVotedOnQuestion = (CurrentUserRecord.ausVotes[nFileQuestion] != 0);
   
      /* If this is the kind of question that the user is choosing from */
      /* right now.                                                     */
      if((bVotedOnQuestion && (nFromWhichQuestions & QUESTIONS_VOTED_ON)) ||
         (!bVotedOnQuestion && (nFromWhichQuestions & QUESTIONS_NOT_VOTED_ON)) ||
         (strcmp(od_control.user_name, QuestionRecord.szCreatorName) == 0
            && (nFromWhichQuestions & QUESTIONS_AUTHORED)))
      {
         /* If question is not deleted. */
         if(!QuestionRecord.bDeleted)
         {
            /* Add one to number of matching questions. */
            ++nQuestions;
         }
      }
      else if((QuestionRecord.bDeleted && (nFromWhichQuestions & QUESTIONS_DELETED)))
      {
         ++nQuestions;
      }
      
      ++nFileQuestion;
   }   
   
   ExclusiveFileClose(fpQuestionFile, hQuestionFile);

   return(nQuestions);
}


int ChooseCategory(int nFromWhichQuestions, char *pszTitle, int *nLocation)
{
}


int ChooseQuestionInCategory(int nCategory, int nFromWhichQuestions,
   char *pszTitle, int *nLocation)
{
}


int ChooseQuestion(int nFromWhichQuestions, char *pszTitle, int *nLocation)
{
   int nCurrent;
   int nFileQuestion = 0;
   int nPagedToQuestion = *nLocation;
   int nDisplayedQuestion = 0;
   char bVotedOnQuestion;
   char chCurrent;
   tQuestionRecord QuestionRecord;
   FILE *fpQuestionFile;
   int hQuestionFile;
   static char szQuestionName[MAX_QUESTIONS][QUESTION_STR_SIZE];
   static int nQuestionNumber[MAX_QUESTIONS];
   BOOL bIsPreviousPage;
   BOOL bIsNextPage;
   
   /* Attempt to open question file. */
   fpQuestionFile = ExclusiveFileOpen(QUESTION_FILENAME, "r+b", &hQuestionFile);

   /* If unable to open question file, assume that no questions have been */
   /* created.                                                            */
   if(fpQuestionFile == NULL)
   {
      /* Display "no questions yet" message. */
      MessageScreen(pszTitle,
         "No questions have been created yet.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      
      /* Indicate that no question has been chosen. */
      return(NO_QUESTION);
   }
   
   /* Loop for every question record in the file. */
   while(fread(&QuestionRecord, sizeof(QuestionRecord), 1, fpQuestionFile) == 1)
   {
      /* Determine whether or not the user has voted on this question. */
      bVotedOnQuestion = (CurrentUserRecord.ausVotes[nFileQuestion] != 0);
      
      /* If this is the kind of question that the user is choosing from */
      /* right now.                                                     */
      if((bVotedOnQuestion && (nFromWhichQuestions & QUESTIONS_VOTED_ON)) ||
         (!bVotedOnQuestion && (nFromWhichQuestions & QUESTIONS_NOT_VOTED_ON)) ||
         (strcmp(od_control.user_name, QuestionRecord.szCreatorName) == 0
            && (nFromWhichQuestions & QUESTIONS_AUTHORED)) ||
         (QuestionRecord.bDeleted && (nFromWhichQuestions & QUESTIONS_DELETED)))
      {
         /* If question is not deleted. */
         if(!QuestionRecord.bDeleted || (nFromWhichQuestions & QUESTIONS_DELETED))
         {
            /* Add this question to list to be displayed. */
            strcpy(szQuestionName[nDisplayedQuestion],
               QuestionRecord.szQuestion);
            nQuestionNumber[nDisplayedQuestion] = nFileQuestion;
         
            /* Add one to number of questions to be displayed in list. */
            nDisplayedQuestion++;
         }
      }
      
      /* Move to next question in file. */
      ++nFileQuestion;
   }   
   
   /* Close question file to allow other nodes to access the file. */
   ExclusiveFileClose(fpQuestionFile, hQuestionFile);

   /* If there are no questions for the user to choose, display an */
   /* appropriate message and return. */
   if(nDisplayedQuestion == 0)
   {
      /* If we were to list all questions. */
      if((nFromWhichQuestions & QUESTIONS_VOTED_ON)
         && (nFromWhichQuestions & QUESTIONS_NOT_VOTED_ON))
      {
         MessageScreen(pszTitle,
            "No questions have been created yet.",
            "Press [Enter] to continue.");
      }
      /* If we were to list questions that the user has voted on. */
      else if(nFromWhichQuestions & QUESTIONS_VOTED_ON)
      {
         MessageScreen(pszTitle,
            "You have not voted on any questions yet.",
            "Press [Enter] to continue.");
      }
      /* If we were to list questions written by current user. */
      else if(nFromWhichQuestions & QUESTIONS_AUTHORED)
      {
         MessageScreen(pszTitle,
            "There are no questions that were created by you.",
            "Press [Enter] to continue.");
      }
      /* Otherwise, we were to list questions that use has not voted on. */
      else if(nFromWhichQuestions & QUESTIONS_NOT_VOTED_ON)
      {
         MessageScreen(pszTitle,
            "You have voted on all of the questions.",
            "Press [Enter] to continue.");
      }
      else if(nFromWhichQuestions & QUESTIONS_DELETED)
      {
         MessageScreen(pszTitle,
            "There are no deleted questions.",
            "Press [Enter] to continue.");
      }
      
      /* Wait for user to press enter key. */
      od_get_answer("\n\r");
      
      /* Return, indicating that no question was chosen. */
      return(NO_QUESTION);
   }

   /* Ensure that initial paged to location is within range. */
   while(nPagedToQuestion >= nDisplayedQuestion)
   {
      nPagedToQuestion -= QUESTION_PAGE_SIZE;
   }

   /* Loop, displaying current page of questions, until the user makes a */
   /* choice.                                                            */
   for(;;)
   {
      /* Display header. */
      ClearWithHeader(pszTitle);
   
      /* Display list of questions on this page. */
      for(nCurrent = 0;
         nCurrent < QUESTION_PAGE_SIZE
         && nCurrent < (nDisplayedQuestion - nPagedToQuestion);
         ++nCurrent)
      {
         /* Determine character to display for current line. */
         if(nCurrent < 9)
         {
            chCurrent = '1' + nCurrent;
         }
         else
         {
            chCurrent = 'A' + (nCurrent - 9);
         }
      
         /* Display this question's title. */
         od_printf("`bright green`%c.`dark green`", chCurrent);
         od_printf(" %s\n\r", szQuestionName[nCurrent + nPagedToQuestion]);
      }

      /* Display prompt for input. */
      od_printf("\n\r`bright white`[Page %d]  Choose a question or:",
         (nPagedToQuestion / QUESTION_PAGE_SIZE) + 1);

      bIsNextPage = nPagedToQuestion < nDisplayedQuestion - QUESTION_PAGE_SIZE;
      bIsPreviousPage = nPagedToQuestion > 0;
      if(bIsNextPage && bIsPreviousPage)
      {
         od_printf(" [N]ext/[P]revious page,");
      }
      else if(bIsNextPage)
      {
         od_printf(" [N]ext page,");
      }
      else if(bIsPreviousPage)
      {
         od_printf(" [P]revious page,");
      }

      if(nFromWhichQuestions & VOTE_ON_ALL)
      {
         od_printf(" [V]ote on all,");
      }

      od_printf(" [Q]uit.");
      
      /* Loop until the user makes a valid choice. */
      for(;;)
      {      
         /* Get input from user */
         chCurrent = toupper(od_get_key(TRUE));
      
         /* Respond to user's input. */
      
         /* If user pressed Q key. */
         if(chCurrent == 'Q')
         {
            /* Return without a choosing a question. */
            return(NO_QUESTION);
         }

         else if(chCurrent == 'V')
         {
            if(nFromWhichQuestions & VOTE_ON_ALL)
            {
               return(ALL_QUESTIONS);
            }
         }
      
         /* If user pressed P key. */
         else if(chCurrent == 'P')
         {
            /* If we are not at the first page. */
            if(nPagedToQuestion > 0)
            {
               /* Move paged to location up one page. */
               nPagedToQuestion -= QUESTION_PAGE_SIZE;
               
               /* Exit user input loop to display next page. */
               break;
            }
         }
      
         /* If user pressed N key. */
         else if(chCurrent == 'N')
         {
            /* If there is more questions after this page. */
            if(nPagedToQuestion < nDisplayedQuestion - QUESTION_PAGE_SIZE)
            {
               /* Move paged.to location down one page. */
               nPagedToQuestion += QUESTION_PAGE_SIZE;

               /* Exit user input loop to display next page. */
               break;
            }
         }
      
         /* Otherwise, check whether the user chose a valid question. */
         else if ((chCurrent >= '1' && chCurrent <= '9')
            || (chCurrent >= 'A' && chCurrent <= 'H'))
         {
            /* Get question number from key pressed. */
            if(chCurrent >= '1' && chCurrent <= '9')
            {
               nCurrent = chCurrent - '1';
            }
            else
            {
               nCurrent = (chCurrent - 'A') + 9;
            }
         
            /* Add current paged to position to user's choice. */
            nCurrent += nPagedToQuestion;

            /* If this is valid question number. */            
            if(nCurrent < nDisplayedQuestion)
            {
               /* Set caller's current question number. */
               *nLocation = nPagedToQuestion;
            
               /* Return actual question number in file. */
               return(nQuestionNumber[nCurrent]);
            }
         }
      }
   }
}


/* The DisplayQuestionResult() function is called to display the results */
/* of voting on a paricular question, and is passed the question record  */
/* of the question. This function is called when the user selects a      */
/* question using the "view results" option, and is also called after    */
/* the user has voted on a question, to display the results of voting on */
/* that question.                                                        */
void DisplayQuestionResult(tQuestionRecord *pQuestionRecord)
{
   int nAnswer;
   int uPercent;

   /* Clear the screen. */
   od_clr_scr();

   /* Check that there have been votes on this question. */
   if(pQuestionRecord->uTotalVotes == 0)
   {
      /* If there have been no votes for this question, display a message */
      /* and return.                                                      */
      MessageScreen("View Results",
         "No one has voted on this qestion yet.",
         "Press [ENTER to continue.");
      od_get_answer("\n\r");
      return;
   }

   /* Display question itself. */
   od_printf("`bright red`%s\n\r", pQuestionRecord->szQuestion);

   /* Display author's name. */
   if(pQuestionRecord->bAnonymous)
   {
      if(strcmp(od_control.sysop_name, od_control.user_name) == 0)
      {
         od_printf("`dark red`Question created `bright flashing red`anonymously`dark red` by %s on %s\n\r",
            pQuestionRecord->szCreatorName,
            ctime(&pQuestionRecord->lCreationTime));
      }
      else
      {
         od_printf("`dark red`Question created anonymously on %s\n\r",
            ctime(&pQuestionRecord->lCreationTime));
      }
   }
   else
   {
      od_printf("`dark red`Question created by %s on %s\n\r",
         pQuestionRecord->szCreatorName,
         ctime(&pQuestionRecord->lCreationTime));
   }
   
   /* Display heading for responses. */
   od_printf("`bright green`Response                        Votes  Percent  Graph\n\r`dark green`");
   if(od_control.user_ansi || od_control.user_avatar)
   {
      od_repeat(196, 79);
   }
   else
   {
      od_repeat('-', 79);
   }
   od_printf("\n\r");

   /* Loop for each answer to the question. */   
   for(nAnswer = 0; nAnswer < pQuestionRecord->nTotalAnswers; ++nAnswer)
   {
      /* Determine percent of users who voted for this answer. */
      uPercent = (pQuestionRecord->auVotesForAnswer[nAnswer] * 100)
         / pQuestionRecord->uTotalVotes;
      
      /* Display answer, total votes and percentage of votes. */
      od_printf("`dark green`%-30.30s  %-5u  %3u%%     `bright white`",
         pQuestionRecord->aszAnswer[nAnswer],
         pQuestionRecord->auVotesForAnswer[nAnswer],
         uPercent);

      /* Display a bar graph corresponding to percent of users who voted */
      /* for this answer.                                                */
      if(od_control.user_ansi || od_control.user_avatar)
      {
         od_repeat(220, (uPercent * 31) / 100);
      }
      else
      {
         od_repeat('=', (uPercent * 31) / 100);
      }

      /* Move to next line. */
      od_printf("\n\r");
   }
   
   /* Display footer. */
   od_printf("`dark green`");
   if(od_control.user_ansi || od_control.user_avatar)
   {
      od_repeat(196, 79);
   }
   else
   {
      od_repeat('-', 79);
   }
   od_printf("\n\r");
   od_printf("`dark green` Total Number of Participants : %u\n\r\n\r",
      pQuestionRecord->uTotalVotes);
   
   /* Wait for user to press enter. */
   WaitForEnter();
}


/* The ReadOrAddCurrentUser() function is used by EZVote to search the    */
/* EZVote user file for the record containing information on the user who */
/* is currently using the door. If this is the first time that the user   */
/* has used this door, then their record will not exist in the user file. */
/* In this case, this function will add a new record for the current      */
/* user. This function returns TRUE on success, or FALSE on failure.      */
int ReadOrAddCurrentUser(void)
{
   FILE *fpUserFile;
   int hUserFile;
   int bGotUser = FALSE;
   int nQuestion;

   /* Attempt to open the user file for exclusive access by this node.     */
   /* This function will wait up to the pre-set amount of time (as defined */   
   /* near the beginning of this file) for access to the user file.        */
   fpUserFile = ExclusiveFileOpen(USER_FILENAME, "a+b", &hUserFile);

   /* If unable to open user file, return with failure. */   
   if(fpUserFile == NULL)
   {
       od_log_write("ERROR 012: Disk error or user file is locked");
       MessageScreen("Error",
         "Unable to access user file. File may be locked.",
         "Press [ENTER] to return to BBS.");
       od_get_answer("\n\r");

      return(FALSE);
   }

   /* Attempt to read the user file header record. */
   if(fread(&UserFileHeader, sizeof(tUserFileHeader), 1, fpUserFile)
      != 1)
   {
      /* If unable to read the user file header record, then create a */
      /* new one and write it. */
      UserFileHeader.unVersionNum = VERSION_NUM;
      UserFileHeader.TimeInstalled = time(NULL);
      UserFileHeader.lUnused = 0;
      UserFileHeader.nMaxQuestions = MAX_QUESTIONS;
      UserFileHeader.nHeaderChecksum = UserFileHeaderChecksum();
      fseek(fpUserFile, 0, SEEK_SET);
      fwrite(&UserFileHeader, sizeof(tUserFileHeader), 1, fpUserFile);
   }
   else
   {
      /* If we did read the user file header record, then check that it */
      /* is valid.                                                      */
      if(UserFileHeader.unVersionNum != VERSION_NUM
         || UserFileHeader.nHeaderChecksum != UserFileHeaderChecksum())
      {
         od_log_write("ERROR 013: User file is damaged or the wrong version");
         MessageScreen("Error",
            "User file is damaged or from the wrong version of EZVote.",
            "Press [ENTER] to return to BBS.");
         od_get_answer("\n\r");
         goto CleanUp;
      }
   }

   /* Begin with the current user record number set to 0. */
   nCurrentUserNumber = 0;

   /* Loop for each record in the file */
   while(fread(&CurrentUserRecord, sizeof(tUserRecord), 1, fpUserFile) == 1)
   {
      /* If name in record matches the current user name ... */
      if(strcmp(CurrentUserRecord.szUserName, od_control.user_name) == 0)
      {
         /* ... then record that we have found the user's record, */
         bGotUser = TRUE;
         
         /* and exit the loop. */
         break;
      }

      /* Move user record number to next user record. */      
      nCurrentUserNumber++;
   }

   /* If the user was not found in the file, attempt to add them as a */
   /* new user if the user file is not already full.                  */
   if(!bGotUser && nCurrentUserNumber < MAX_USERS)
   {
      /* Place the user's name in the current user record. */
      strcpy(CurrentUserRecord.szUserName, od_control.user_name);

      /* Reset unused field. */
      CurrentUserRecord.lUnused = 0;
      
      /* Record that user hasn't voted on any of the questions. */
      for(nQuestion = 0; nQuestion < MAX_QUESTIONS; ++nQuestion)
      {
         CurrentUserRecord.ausVotes[nQuestion] = 0x0000;
      }
      
      /* Write the new record to the file. */
      if(fwrite(&CurrentUserRecord, sizeof(tUserRecord), 1, fpUserFile) == 1)
      {
         /* If write succeeded, record that we now have a valid user record. */
         bGotUser = TRUE;
      }
   }
   else if(nCurrentUserNumber >= MAX_USERS)
   {
      MessageScreen("Error",
         "User file is full. No new users may enter EZVote.",
         "Press [ENTER] to return to BBS.");
      od_get_answer("\n\r");
      od_log_write("ERROR 014: User file is full");
   }

CleanUp:
   /* Close the user file to allow other nodes to access it. */
   ExclusiveFileClose(fpUserFile, hUserFile);

   /* Return, indciating whether or not a valid user record now exists for */
   /* the user that is currently online.                                   */   
   return(bGotUser);
}


/* The WriteCurrentUser() function is called to save the information on the */
/* user who is currently using the door, to the EZVUSER.DAT file.            */
void WriteCurrentUser(void)
{
   FILE *fpUserFile;
   int hUserFile;

   /* Attempt to open the user file for exclusive access by this node.     */
   /* This function will wait up to the pre-set amount of time (as defined */   
   /* near the beginning of this file) for access to the user file.        */
   fpUserFile = ExclusiveFileOpen(USER_FILENAME, "r+b", &hUserFile);

   /* If unable to access the user file, display an error message and */
   /* return.                                                         */
   if(fpUserFile == NULL)
   {
      od_log_write("ERROR 015: Disk error or user file locked.");
      MessageScreen("Error",
         "Unable to access the user file to update your information.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      return;
   }
   
   /* Move to appropriate location in user file for the current user's */
   /* record. */
   fseek(fpUserFile, sizeof(tUserFileHeader) +
      ((long)nCurrentUserNumber * sizeof(tUserRecord)), SEEK_SET);

   /* Write the new record to the file. */
   if(fwrite(&CurrentUserRecord, sizeof(tUserRecord), 1, fpUserFile) == 1)
   {
      /* If unable to write the record, display an error message. */
      ExclusiveFileClose(fpUserFile, hUserFile);
      od_log_write("ERROR 016: Unable to write to user file.");
      MessageScreen("Error",
         "Unable to access the user file to update your information.",
         "Press [Enter] to continue.");
      od_get_answer("\n\r");
      return;
   }
   
   /* Close the user file to allow other nodes to access it again. */
   ExclusiveFileClose(fpUserFile, hUserFile);
}




/* This function is used by Vote to open a file. If Vote has been compiled */
/* with #define MULTINODE_AWARE uncommented (see the beginning of this     */
/* file), file access is performed in a multinode-aware way. This implies  */
/* that the file is opened of exclusive access, using share-aware open     */
/* functions that may not be available using all compilers.                */
FILE *ExclusiveFileOpen(char *pszFileName, char *pszMode, int *phHandle)
{
#ifdef MULTINODE_AWARE
   /* If Vote is being compiled for multinode-aware file access, then   */
   /* attempt to use compiler-specific share-aware file open functions. */
   FILE *fpFile = NULL;
   time_t StartTime = time(NULL);
   int hFile;

   /* Attempt to open the file while there is still time remaining. */    
   while((hFile = sopen(pszFileName, O_BINARY | O_RDWR, SH_DENYRW,
      S_IREAD | S_IWRITE)) == -1)
   {
      /* If we have been unable to open the file for more than the */
      /* maximum wait time, or if open failed for a reason other   */
      /* than file access, then attempt to create a new file and   */
      /* exit the loop.                                            */
      if(errno != EACCES ||
         difftime(time(NULL), StartTime) >= FILE_ACCESS_MAX_WAIT)
      {
         hFile = sopen(pszFileName, O_BINARY | O_CREAT, SH_DENYRW,
            S_IREAD | S_IWRITE);
         break;
      }

      /* If we were unable to open the file, call od_kernel, so that    */
      /* OpenDoors can continue to respond to sysop function keys, loss */
      /* of connection, etc.                                            */
      od_kernel();
   }

   /* Attempt to obtain a FILE * corresponding to the handle. */
   if(hFile != -1)
   {
      fpFile = fdopen(hFile, pszMode);
      if(fpFile == NULL)
      {
         close(hFile);
      }
   }

   /* Pass file handle back to the caller. */
   *phHandle = hFile;

   /* Return FILE pointer for opened file, if any. */   
   return(fpFile);
#else
   /* Ignore unused parameters. */
   (void)phHandle;

   /* If Vote is not being compiled for multinode-aware mode, then just */
   /* use fopen to access the file.                                     */
   return(fopen(pszFileName, pszMode));
#endif
}


/* The ExclusiveFileClose() function closes a file that was opened using */
/* ExclusiveFileOpen().                                                  */
void ExclusiveFileClose(FILE *pfFile, int hHandle)
{
   fclose(pfFile);
#ifdef MULTINODE_AWARE
   close(hHandle);
#else
   /* Ignore unused parameters. */
   (void)hHandle;
#endif
}


/* The WaitForEnter() function is used by EZVote to create its custom */
/* "Press [ENTER] to continue." prompt.                               */
void WaitForEnter(void)
{
   /* Display prompt. */
   od_printf("`bright white`Press [ENTER] to continue.\n\r");
   
   /* Wait for a Carriage Return or Line Feed character from the user. */
   od_get_answer("\n\r");
}


int UserFileHeaderChecksum(void)
{
   int *paHeader = (int *)&UserFileHeader;
   return(paHeader[0] + paHeader[1] - paHeader[2] + paHeader[3]
      - paHeader[4] + paHeader[5]);
}


void NoDropFileFunc(void)
{
   od_control.od_force_local = TRUE;
}


BOOL ReadCategoryInfo(tCategoryInfo *pCategoryInfo)
{
   tQuestionRecord QuestionRecord;
   tCategoryInfoRecord *pCategoryRecord;
   FILE *fpQuestionFile;
   int hQuestionFile;
   int nQuestion = 0;

   FreeCategoryArray(pCategoryInfo);

   /* Attempt to open question file. */
   fpQuestionFile = ExclusiveFileOpen(QUESTION_FILENAME, "r+b", &hQuestionFile);

   /* If unable to open question file, assume that no questions have been */
   /* created.                                                            */
   if(fpQuestionFile == NULL)
   {
      return(TRUE);
   }

   /* Loop for every question record in the file. */
   while(fread(&QuestionRecord, sizeof(QuestionRecord), 1, fpQuestionFile) == 1)
   {
      /* Determine whether or not the user has voted on this question. */
      BOOL bVotedOnQuestion = (CurrentUserRecord.ausVotes[nQuestion] != 0);

      /* Get category record for this question. */
      pCategoryRecord = GetCategoryInfoRecord(&CategoryInfo,
         QuestionRecord.szCategory);

      if(pCategoryRecord == NULL) return(FALSE);

      /* Update the information in this category. */
      if(QuestionRecord.bDeleted)
      {
         pCategoryRecord->bAnyDeletedInCategory = TRUE;
      }
      else
      {
         if(bVotedOnQuestion)
         {
            pCategoryRecord->bAnyVotedInCategory = TRUE;
         }
         else
         {
            pCategoryRecord->bAnyNotVotedInCategory = TRUE;
         }

         if(strcmp(od_control.user_name, QuestionRecord.szCreatorName) == 0)
         {
            pCategoryRecord->bAnyAuthoredInCategory = TRUE;
         }
      }

      /* Move to the next question. */      
      ++nQuestion;
   }

   /* Close exclusively opened file. */   
   ExclusiveFileClose(fpQuestionFile, hQuestionFile);

   SortCategoryInfoRecords(pCategoryInfo);

   return(TRUE);
}


void FreeCategoryArray(tCategoryInfo *pCategoryInfo)
{
   if(pCategoryInfo->paCategoryRecord != NULL)
   {
      free(pCategoryInfo->paCategoryRecord);
      pCategoryInfo->paCategoryRecord = NULL;
   }

   pCategoryInfo->nTotalCategories = 0;
   pCategoryInfo->nCategoryArraySize = 0;
}


tCategoryInfoRecord *AddCategoryRecord(tCategoryInfo *pCategoryInfo)
{
   /* Grow the category array if needed. */
   if(pCategoryInfo->nTotalCategories == pCategoryInfo->nCategoryArraySize)
   {
      int nNewArraySize = pCategoryInfo->nCategoryArraySize +
         GROW_CATEGORY_ARRAY_BY;

      tCategoryInfoRecord *paNewCategoryArray =
         realloc(pCategoryInfo->paCategoryRecord,
                  nNewArraySize * sizeof(tCategoryInfoRecord));

      if(paNewCategoryArray == NULL)
      {
         return(NULL);
      }

      pCategoryInfo->paCategoryRecord = paNewCategoryArray;
      pCategoryInfo->nCategoryArraySize = nNewArraySize;
   }

   memset(&pCategoryInfo->paCategoryRecord[pCategoryInfo->nTotalCategories],
          0, sizeof(tCategoryInfoRecord));

   return(&pCategoryInfo->paCategoryRecord[pCategoryInfo->nTotalCategories++]);
}


tCategoryInfoRecord *GetCategoryInfoRecord(tCategoryInfo *pCategoryInfo,
                                           char *pszCategory)
{
   int n;
   tCategoryInfoRecord *pNewRecord;

   /* Loop through the array of categories, looking for a match. */
   for(n = 0; n < pCategoryInfo->nTotalCategories; ++n)
   {
      if(strcmp(pCategoryInfo->paCategoryRecord[n].szCategory, pszCategory)
         == 0)
      {
         return(&pCategoryInfo->paCategoryRecord[n]);
      }
   }

   /* If we got here, then we didn't find a match, so try to add a new one. */
   pNewRecord = AddCategoryRecord(pCategoryInfo);
   if(pNewRecord == NULL)
   {
      return(NULL);
   }

   strcpy(pNewRecord->szCategory, pszCategory);

   return(pNewRecord);
}


void SortCategoryInfoRecords(tCategoryInfo *pCategoryInfo)
{
   qsort(&pCategoryInfo->paCategoryRecord,
         pCategoryInfo->nCategoryArraySize,
         sizeof(tCategoryInfoRecord),
         CompareCategoryRecords);
}


int CompareCategoryRecords(const void *pElement1, const void *pElement2)
{
   tCategoryInfoRecord *pRecord1 = (tCategoryInfoRecord *)pElement1;
   tCategoryInfoRecord *pRecord2 = (tCategoryInfoRecord *)pElement2;

   return(strcmp(pRecord1->szCategory, pRecord2->szCategory));
}


BOOL CompactDatabase()
{
   tQuestionRecord QuestionRecord;
   tUserRecord UserRecord;
   FILE *fpOldQuestionFile;
   int hOldQuestionFile;
   FILE *fpOldUserFile;
   int hOldUserFile;
   FILE *fpNewQuestionFile;
   int hNewQuestionFile;
   FILE *fpNewUserFile;
   int hNewUserFile;
   int nQuestion = 0;
   char abDeleted[MAX_QUESTIONS];
   int nFrom;
   int nTo;
   int nTotalQuestions;
   
   /* Attempt to open old question file. */
   fpOldQuestionFile = ExclusiveFileOpen(QUESTION_FILENAME, "r+b",
      &hOldQuestionFile);
   if(fpOldQuestionFile == NULL)
   {
      return(FALSE);
   }

   /* Attempt to open old user file. */
   fpOldUserFile = ExclusiveFileOpen(USER_FILENAME, "r+b", &hOldUserFile);
   if(fpOldUserFile == NULL)
   {
      ExclusiveFileClose(fpOldQuestionFile, hOldQuestionFile);
      return(FALSE);
   }

   /* Create new question file. */
   unlink(TEMP_QUESTION_FILENAME);
   fpNewQuestionFile = ExclusiveFileOpen(TEMP_QUESTION_FILENAME, "r+b",
      &hNewQuestionFile);
   if(fpNewQuestionFile == NULL)
   {
      ExclusiveFileClose(fpOldUserFile, hOldUserFile);
      ExclusiveFileClose(fpOldQuestionFile, hOldQuestionFile);
      return(FALSE);
   }

   /* Create new user file. */
   unlink(TEMP_USER_FILENAME);
   fpNewUserFile = ExclusiveFileOpen(TEMP_USER_FILENAME, "r+b", &hNewUserFile);
   if(fpNewUserFile == NULL)
   {
      ExclusiveFileClose(fpNewQuestionFile, hNewQuestionFile);
      ExclusiveFileClose(fpOldUserFile, hOldUserFile);
      ExclusiveFileClose(fpOldQuestionFile, hOldQuestionFile);
      return(FALSE);
   }
   
   /* Loop for every question record in the question file. */
   while(fread(&QuestionRecord, sizeof(QuestionRecord), 1, fpOldQuestionFile) == 1)
   {
      abDeleted[nQuestion] = QuestionRecord.bDeleted;

      /* Copy this record if and only if we have success. */
      if(!QuestionRecord.bDeleted)
      {
         fwrite(&QuestionRecord, sizeof(QuestionRecord), 1, fpNewQuestionFile);
      }
   
      ++nQuestion;
   }

   nTotalQuestions = nQuestion;

   /* Copy header record from old user file to new user file. */
   fread(&UserFileHeader, sizeof(tUserFileHeader), 1, fpOldUserFile);
   fwrite(&UserFileHeader, sizeof(tUserFileHeader), 1, fpNewUserFile);

   /* Loop for every user record in the user file. */
   while(fread(&UserRecord, sizeof(UserRecord), 1, fpOldUserFile) == 1)
   {
      /* Update the user's vote information. */
      for(nFrom = 0, nTo = 0; nFrom < nTotalQuestions; ++nFrom)
      {
         if(!abDeleted[nFrom])
         {
            if(nFrom != nTo)
            {
               UserRecord.ausVotes[nTo] = UserRecord.ausVotes[nFrom];
            }

            ++nTo;
         }
      }

      /* Zero-out remaining vote information. */
      for(;nTo < MAX_QUESTIONS; ++nTo)
      {
         UserRecord.ausVotes[nTo] = 0;
      }

      /* Write updated user record back to the file. */
      fwrite(&UserRecord, sizeof(UserRecord), 1, fpNewUserFile);
   }

   /* Close both old and new files. */
   ExclusiveFileClose(fpNewUserFile, hNewUserFile);
   ExclusiveFileClose(fpNewQuestionFile, hNewQuestionFile);
   ExclusiveFileClose(fpOldUserFile, hOldUserFile);
   ExclusiveFileClose(fpOldQuestionFile, hOldQuestionFile);

   /* Replace old files with new files. */
   unlink(USER_FILENAME);
   unlink(QUESTION_FILENAME);
   rename(TEMP_USER_FILENAME, USER_FILENAME);
   rename(TEMP_QUESTION_FILENAME, QUESTION_FILENAME);

   /* Compaction is now completed, so return with success. */
   return(TRUE);
}


void SysopCommands()
{
   char chMenuChoice;
   int nUndeletedQuestions;
   int nDeletedQuestions;
   int nUsedRecords;
   int nMaxQuestions;
   int nPercentOfFileUsed;
   int nSavedByCompaction;

   do
   {
      /* Update statistics. */
      nUndeletedQuestions = NumQuestions(QUESTIONS_NOT_DELETED);
      nDeletedQuestions = NumQuestions(QUESTIONS_DELETED);
      nUsedRecords = nUndeletedQuestions + nDeletedQuestions;
      nMaxQuestions = MAX_QUESTIONS;
      nPercentOfFileUsed = (nUsedRecords * 100) / nMaxQuestions;
      nSavedByCompaction = (nDeletedQuestions * 100) / nMaxQuestions;

      if(bUseOldUI)
      {
         ClearWithHeader("Sysop Menu");

         od_printf("`dark green`\n\r");
         od_printf("       Number of active questions : `bright green`%d`dark green`\n\r",
            nUndeletedQuestions);
         od_printf("      Number of deleted questions : `bright green`%d`dark green`\n\r",
            nDeletedQuestions);
         od_printf("   Percent of database space used : `bright green`%d%%`dark green`\n\r",
            nPercentOfFileUsed);
         od_printf("      Percent compaction can save : `bright green`%d%%`dark green`\n\r\n\r\n\r",
            nSavedByCompaction);

         od_printf("        [`bright green`A`dark green`] Add a new question\n\r\n\r");

         if(bUseCategories)
         {
            od_printf("        [`bright green`M`dark green`] Move question to a different category\n\r\n\r");
         }

         od_printf("        [`bright green`D`dark green`] Delete a question\n\r");
         od_printf("        [`bright green`U`dark green`] Undelete a deleted question\n\r\n\r");

         od_printf("        [`bright green`C`dark green`] Compact question database\n\r\n\r");

         od_printf("        [`bright green`Q`dark green`] Quit to main menu\n\r");

         chMenuChoice = od_get_answer("AMDUCQ");
      }
      else
      {
         switch(od_popup_menu("Sysop Menu",
            "^Add a new question|^Move question to a different category|^Delete a question|^Undelete a deleted question|^Compact question database",
            40, 13, 1, MENU_KEEP | MENU_ALLOW_CANCEL))
         {
            case 0:
               chMenuChoice = 'Q';
               break;
            case 1:
               chMenuChoice = 'A';
               break;
            case 2:
               chMenuChoice = 'M';
               break;
            case 3:
               chMenuChoice = 'D';
               break;
            case 4:
               chMenuChoice = 'U';
               break;
            case 5:
               chMenuChoice = 'C';
               break;
         }
      }


      switch(chMenuChoice)
      {
         case 'A':
            AddQuestion();
            break;

         case 'M':
            if(bUseCategories)
            {
            }
            break;

         case 'D':
            DeleteOrUndeleteQuestion(TRUE);
            break;

         case 'U':
            DeleteOrUndeleteQuestion(FALSE);
            break;

         case 'C':
            ClearWithHeader("Compact Database");

            od_printf("\n\r");
            od_printf("Compaction permanently removes deleted questions from the question database.\n\r");
            od_printf("By compacting your EZVote question database from time to time, you will make\n\r");
            od_printf("more space available for additional questions, and save hard disk space.\n\r");
            od_printf("However, once compaction has been performed, no previously deleted questions\n\r");
            od_printf("may be undeleted.\n\r\n\r");

            od_printf("`bright flashing green`*** WARNING ***`dark green` Database compaction must only be performed when no other users\n\r");
            od_printf("                are using EZVote. Proceeding with compaction while other\n\r");
            od_printf("                nodes are active may result in database corruption.\n\r\n\r");

            if(nDeletedQuestions == 0)
            {
               od_printf("There are currently no deleted questions, so database compaction is not\n\r");
               od_printf("necessary at this time.\n\r\n\r");

               WaitForEnter();

               break;
            }

            od_printf("`bright white`Do you wish to proceed with database compaction at this time? (Y/N)`dark green`");

            if(od_get_answer("YN") == 'Y')
            {
               od_printf("\n\r\n\r");
               od_printf("Now compacting EZVote question database ...");

               CompactDatabase();

               od_printf(" done!\n\r\n\r");

               od_printf("Your EZVote question database has now been compacted. %d deleted questions\n\r");
               od_printf("were removed.\n\r\n\r");

               WaitForEnter();
            }
            break;
      }

   } while(chMenuChoice != 'Q');
}
