{$A+,B-,D+,E+,F+,G-,I-,L+,N-,O-,R-,S+,V+,X+}
{$M 16384,0,655360}
Unit StrMisc; { by John Stephenson; Copyright 1995 }
{ Arne de.Bruijn wrote DayOfWeek, and part of DayLightSavings }
Interface
Uses Asmmisc,Dos;
Const
  JustChar: char = ' '; { Character to do justification & centering with }

Function AddBackSlash(st: PathStr): PathStr;
Function AttrToStr(attr: word): string;
Function BinToHunk(st: string): string;
Function BooleanToStr(b: boolean): String;
Function CarretProcess(st: string): String; { "^M"'s => #13's }
Function Center(St: String; maxplace: byte): String;
Function CharsInStr(Ch: Char; St: String): byte;
Function CommaStr(Num: Longint): String;
Function Del(s: String; count,index: byte): string;
Function DtToDateStr(dt: datetime): String;
Function DtToTimeStr(dt: datetime): String;
Function Dup(ch: char; times: byte): String;
Function Getword(st: string; whichword: byte): string;
Function HexToInt(HexStr: String): Longint;
Function HunkToBin(st: string): string;
Function Ins(source,s: String; index: byte): string;
Function InString(st: String; ch: char; times: byte): boolean;
Function IntToBin(Num: longint; digits: byte): String;
Function IntToDayStr(Day: integer): String;
Function IntToHex(Num: longint; digits: byte): String;
Function IntToMonthStr(Month: integer): String;
Function IntToStr(Num: longint): String;
Function IsLetter(ch: char): boolean;
Function IsNumber(Ch: char): boolean;
Function LJust(St: String; Maxplace: Byte): String;
Function LTrim(St: String): String;
Function LZero(Num: Longint; Zeros: Byte): String;
Function LastPos(Ch: char; St: string): byte;
Function MetricStr(Num: Longint): String;
Function Proper(St: String): String;
Function RJust(St: String; Maxplace: Byte): String;
Function RTrim(St: String): String;
Function RemoveBackSlash(st: PathStr): PathStr;
Function StrToInt(St: String): longint;
Function TotalWords(St: String): byte;
Function Trim(St: String): String;
Function WordWrap(var st: string; place: byte): string;
Procedure DateStrToDt(St: String; var Dt: DateTime);
Procedure ReverseStr(var st: string);
Procedure TimeStrToDt(St: String; var Dt: DateTime);

{ Date features }
Function DayLightSavings(dt: datetime): boolean;
Function LeapYear(year: integer): boolean;
Function DayOfWeek(Year,Month,Day: Integer): Byte;
Procedure GregorianToJulian(Year,Month,Day: word; Var Julian: LongInt);
Procedure JulianToGregorian(Julian: LongInt; Var Year,Month,Day: Word);
Procedure PackUnixTime(dt: datetime; var uts: longint);
Procedure UnPackUnixTime(uts: longint; Var dt: datetime);

Implementation
Const { Used for conversions of dates }
  D0    =    1461;
  D1    =  146097;
  D2    = 1721119;

  hexid: array[$00..$0F] of char = '0123456789ABCDEF'; { For dec to hex }

Function AttrToStr(attr: word): string;
{ Turns an attribute value into a string, containing if it has the readonly,
  system, hidden, attribute, directory or volumeid settings active }
var result: string[5];
begin
  result := dup('_',5);
  if attr and readonly>0 then result[1] := 'R';
  if attr and hidden>0 then result[2] := 'H';
  if attr and sysfile>0 then result[3] := 'S';
  if attr and archive>0 then result[4] := 'A';
  if attr and directory>0 then result[5] := 'D'
  else if attr and volumeid>0 then result[5] := 'V';
  AttrToStr := result;
end;

Function Ins(source,s: String; index: byte): string;
{ Inserts the string source, into the string s at position index, then
  returns the result }
begin
  Insert(Source,S,Index);
  Ins := s;
end;

Function Del(s: String; count,index: byte): string;
{ Deletes count characters from string s, at position index, and returns
  the result }
begin
  Delete(S,index,count);
  Del := s;
end;

Function BooleanToStr(b: boolean): String;
{ Does not produce the same "TRUE" "FALSE" as pascal, but "True" "False" }
begin
  if b then booleantostr := 'True'
  else booleantostr := 'False';
end;

Function IntToDayStr(Day: integer): String;
{ Turns an integer into it's appropriate day of the week string }
begin
  case day of
    0: IntToDayStr := 'Sunday';
    1: IntToDayStr := 'Monday';
    2: IntToDayStr := 'Tuesday';
    3: IntToDayStr := 'Wednesday';
    4: IntToDayStr := 'Thursday';
    5: IntToDayStr := 'Friday';
    6: IntToDayStr := 'Saturday';
    else inttodaystr := '';
  end;
end;

Function IntToMonthStr(Month: integer): String;
{ Turns an integer into it's appropriate month }
begin
  case month of
    1: IntToMonthStr := 'January';
    2: inttoMonthstr := 'February';
    3: inttoMonthstr := 'March';
    4: inttoMonthstr := 'April';
    5: IntToMonthStr := 'May';
    6: inttoMonthstr := 'June';
    7: inttoMonthstr := 'July';
    8: inttoMonthstr := 'August';
    9: IntToMonthStr := 'September';
    10: inttoMonthstr := 'October';
    11: inttoMonthstr := 'November';
    12: inttoMonthstr := 'December';
    else inttoMonthstr := '';
  end;
end;

Function DtToDateStr(dt: datetime): String;
{ This function converts a dt into a date string, in the format of
  MM/DD/YYYY. }
begin
  with dt do
    DtToDateStr := inttostr(month)+'/'+lzero(day,2)+'/'+inttostr(year);
end;

Function DtToTimeStr(dt: datetime): String;
{ This converts a DT into the time field. It is done in the military
  format, ie ??:??:?? (starting at 00:00:00, ending at 23:59:59) }
begin
  with dt do
    DtToTimeStr := inttostr(hour)+':'+lzero(min,2)+':'+lzero(sec,2);
end;

Procedure TimeStrToDt(st: string; var dt: datetime);
{ Converts a time in the military format ??:??:?? or ??:??.?? into the
  time fields for DT (starting at 00:00:00, ending at 23:59:59). }
var place: byte; code: word;
begin
  while pos('.',st)>0 do st[pos('.',st)] := ':';
  place := pos(':',st);
  val(copy(st,1,place-1),dt.hour,code);
  delete(st,1,place);

  place := pos(':',st);
  val(copy(st,1,place-1),dt.min,code);
  delete(st,1,place);

  val(st,dt.sec,code);
end;

Procedure DateStrToDt(st: string; var dt: datetime);
{ This function converts a string "st" into the date fields of variable
  dt. The expected format to be recieved is either: MM/DD/YYYY or
  MM/DD/YY. You also can use dots, or hyphens to seperate the fields. }
var place: byte; code: word;
begin
  while pos('.',st)>0 do st[pos('.',st)] := '/';
  while pos('-',st)>0 do st[pos('-',st)] := '/';

  place := pos('/',st);
  val(copy(st,1,place-1),dt.month,code);
  delete(st,1,place);

  place := pos('/',st);
  val(copy(st,1,place-1),dt.day,code);
  delete(st,1,place);

  val(st,dt.year,code);
  { Check to see if the current century part was put in, if not then add  
    it to the current date set. }
  if dt.year < 100 then dt.year := 1900+dt.year;
End;

Function CharsInStr(Ch: Char; St: String): byte;
{ Returns the number of times the character "ch" is repeated in the string
  "st" }
var chars: byte;
begin
  chars := 0;
  While st<>'' do begin
    if ch = st[1] then inc(chars);
    delete(st,1,1);
  end;
  CharsInStr := Chars;
end;

Function TotalWords(St: String): byte;
{ This returns the total words seperated by the character "justchar"
  in the string "st" }
begin
  if length(st) = 0 then totalwords := 0
  else totalwords := charsinstr(justchar,st)+1;
end;

Function Getword(st: string; whichword: byte): String;
{ Get word number "whichword" in string "st". Use the justchar instead of
  using literal spaces in the searching, ie so you can find words in
  comma'd strings. Eg NodeLists use ","'s instead of spaces to seperate
  fields/words. }
Var
  Place,loop: byte;
  Temp: string;
Begin
  { Delete double spaces that may interfere with getting each word }
  While (pos('  ',st)<>0) do delete(st,pos('  ',st),1);
  Place := 1;
  Temp := '';
  For loop := 1 to whichword - 1 do begin
    While st[place] <> justchar do inc(place);
    While st[place] = justchar do inc(place);
  End;
  While (st[place] <> justchar) and (place <= byte(st[0])) do begin
    Temp := temp + st[place];
    Inc(place);
  End;
  Getword := temp;
End;

Function CarretProcess(st: String): String;
{ This function takes "st" and takes all the characters such as '^M', '^H',
  '^J' in the stirng and turns them into ^M, ^H, ^J (literal) }
var return: string;
begin
  return := '';
  while st <> '' do begin
    case st[1] of
      '^': begin
        st[2] := upcase(st[2]);
        return := return+char(byte(st[2])-byte('A')+1);
        delete(st,1,2);
      End;
      else begin
        return := return+st[1];
        delete(st,1,1);
      End;
    End;
  End;
  CarretProcess := return;
End;

Function IntToStr(Num: longint): String;
{ This function takes an integer value, and creates a string }
Var st: string;
Begin
  Str(Num,St);
  IntToStr := st;
End;

Function StrToInt(St: String): longint;
{ String to integer value }
Var num: longint; code: integer;
Begin
  Val(St,num,code);
  StrToInt := num;
End;

Function HexToInt(HexStr: String): Longint;
{ Hexadecimal to integer value }
Var
  I,HexNibble: word;
  Temp: Longint;
  Code: integer;
Begin
  Temp := 0;
  hexstr := upcasestr(hexstr);

  { Remove all of the garbage characters, ie $3F8, or 3F8h
    MUST be done in reverse order, as not to mess up the indexes in the
    string with the old ones! }
  for i := length(hexstr) downto 1 do
    if not (HexStr[i] in ['0'..'9','A'..'F']) then delete(hexstr,i,1);

  { Most significant on left, right? So do it from the right where it's
    accending in order to save headaches }
  For I := Length(HexStr) downto 1 do Begin
    If HexStr[I] in ['0'..'9'] then hexnibble := Byte(HexStr[i]) - byte('0')
    else hexnibble := Byte(HexStr[i]) - byte('A')+10;

    Inc(Temp,longint(HexNibble) * (1 shl (4*(longint(Length(HexStr)) - I))));
  End;
  HexToInt := Temp;
End;

Function IntToHex(num: longint; digits: byte): String;
var
  s: String;
  c: byte;
  n: array[1..sizeof(longint)] of byte absolute num;
begin
  s := '';
  for c := 4 downto 1 do s := s + hexid[n[c] shr 4]+hexid[n[c] and $F];
  IntToHex := copy(s,8-digits+1,digits);
End;

Function IntToBin(Num: longint; digits: byte): String;
var Result : String;
begin
  Result := '';
  while (num > 0) do begin
    Result := Char(num mod 2+Byte('0'))+result;
    Num := Num div 2;
  end;
  if (Result = '') then Result := '0';
  while length(result) < digits do result := '0'+result;
  IntToBin := Result;
end;

Function LZero(Num: longint; Zeros: Byte): String;
{ Justifies the num into places, padding with zeros }
Var st: String;
Begin
  Str(Num,St);
  while Length(st) < Zeros do St := '0' + St;
  LZero := St;
End;

Function LJust(St: String; Maxplace: Byte): String;
{ Appends spaces until justified to maxplace }
Begin
  While MaxPlace > length(st) do st := st+justchar;
  Ljust := copy(st,1,maxplace);
End;

Function RJust(St: String; Maxplace: Byte): String;
{ Appends to begining spaces until justified to maxplace }
Begin
  While MaxPlace > length(st) do st := justchar+st;
  Rjust := copy(st,1,maxplace);
End;

Function InString(st: string; ch: char; times: byte): boolean;
{ If "ch" is in the string "st" "times" times then this function will
  return true, good for checking to insure a date field (MM/DD/YYYY)
  contains the required two /'s and not more or less }
Var place: byte;
Begin
  repeat
    dec(times);
    place := pos(ch,st);
    if place <> 0 then delete(st,place,1);
  until (place = 0) or (times = 0);

  if place = 0 then instring := false
  else begin
    instring := pos(ch,st) = 0;
  end;
end;

Function Dup(ch: char; times: byte): String;
{ Dups Ch "times" and returns, good for things like: "---------------" }
Var temp: String;
begin
  fillchar(temp[1],times,ch);
  temp[0] := char(times);
  dup := temp;
End;

Function LastPos(Ch: char; St: string): byte;
{ Just like Pos, except in reverse, and this only works with characters
  instead of supporting both strings in the substring field }
Var place: byte;
Begin
  Place := succ(Length(St));
  Repeat
    Dec(Place);
  Until (Place = 0) or (St[place] = Ch);
  LastPos := Place;
End;

Function WordWrap(var st: string; place: byte): string;
{ Take the string (st) and return up to desired "place" of it, and move
  back to space (justchar), and finally delete what was in the string.
  And do it in as few lines as possible. PHEW! }
Var TruncAt: byte;
begin
  if place <= length(st) then begin
    TruncAt := lastpos(justchar,copy(st,1,place));
    If TruncAt = 0 then truncat := place;
  End
  Else truncat := length(st);
  WordWrap := copy(st,1,truncat);
  Delete(st,1,truncat);
End;

Function IsNumber(Ch: char): boolean;
{ Is ch a valid number? }
begin
  Isnumber := ch in ['0'..'9'];
end;

Function IsLetter(ch: char): boolean;
{ Is ch a valid letter? }
begin
  Isletter := upcase(ch) in ['A'..'Z'];
end;

Function Center(St: String; maxplace: byte): String;
{ Center the string to fit inbetween maxplace }
var temp: string; num: byte;
Begin
  num := (maxplace div 2)-(length(st) div 2);
  temp := dup(justchar,num);
  temp := temp+st;
  temp := temp+dup(justchar,maxplace-num-length(st));
  center := temp;
End;

Function Proper(St: String): String;
{ Properly capitalize a string. Ie "john" would become "John" and "JOHN"
  would become "John". To do this, we must first convert to lowercase,
  then uppercase the first letter, and any characters following each
  justification character (space). }
Var loop: byte;
begin
  st := locasestr(st);
  st[1] := upcase(st[1]);
  for loop := 1 to length(st) do if (st[loop] = justchar) then
    st[loop+1] := upcase(st[loop+1]);
  Proper := st;
end;

Function RTrim(St: String): String;
{ Trim all justification characters (spaces) on the right side }
Begin
  while (st[0] <> #0) and (st[length(st)] = justchar) do dec(st[0]);
  Rtrim := st;
End;

Function LTrim(St: String): String;
{ Trim all the justification characters (spaces) on the left side }
Begin
  while (st[0] <> #0) and (st[1] = justchar) do delete(st,1,1);
  Ltrim := st;
end;

Function Trim(St: String): String;
{ Trim all the justification characters (spaces) on both the sides }
begin
  while (st[0] <> #0) and (st[length(st)] = justchar) do dec(st[0]);
  while (st[0] <> #0) and (st[1] = justchar) do delete(st,1,1);
  Trim := st;
end;

Procedure ReverseStr(var st: string);
{ Reverses "str" }
Var
  temp: string;
  loop: byte;
Begin
  temp[0] := st[0];
  for loop := 1 to length(st) do temp[length(st)-loop+1] := st[loop];
  st := temp;
End;

Function CommaStr(Num: Longint): String;
{ To "comma" a string, using the obsolete imperial standard. }
Var
  loop: byte;
  st: string;
  times: shortint;
Begin
  str(num,st);
  reversestr(st);
  times := length(st) div 3;
  if length(st) mod 3 = 0 then dec(times);
  for loop := 1 to times do insert(',',st,loop*3+loop);
  reversestr(st);
  commastr := st;
End;

Function MetricStr(Num: Longint): String;
{ Do NOT put spaces in if it's just 1000-9999 (ie <4 in length), only
  10000+ according to metric standard, and then every 3 character. }
Var
  loop: byte;
  st: string;
  times: shortint;
Begin
  str(num,st);
  if length(st) > 4 then begin
    reversestr(st);
    times := length(st) div 3;
    if length(st) mod 3 = 0 then dec(times);
    for loop := 1 to times do insert(' ',st,loop*3+loop);
    reversestr(st);
  end;
  metricstr := st;
End;

Procedure GregorianToJulian(Year,Month,Day: word; Var Julian: LongInt);
{ Converts a Gregorian date (standard date) to a Julian date }
Var
  Century,
  XYear: LongInt;
Begin
  If Month <= 2 then begin
    Dec(Year);
    Inc(month,12);
  End;
  dec(Month, 3);
  Century := Year div 100;
  XYear := Year mod 100;
  Century := (Century * D1) shr 2;
  XYear := (XYear * D0) shr 2;
  Julian := ((((Month*153)+2) div 5)+Day)+D2+XYear+Century;
End;

Procedure JulianToGregorian(Julian: LongInt; Var Year,Month,Day: Word);
{ Converts a Julian date, to a Gregorian date (standard date) }
Var
  Temp, XYear: LongInt;
  YYear, YMonth, YDay : Integer;
begin
  Temp := (((Julian - D2) shl 2) - 1);
  XYear := (Temp mod D1) or 3;
  Julian := Temp div D1;
  YYear := (XYear div D0);
  Temp := ((((XYear mod D0) + 4) shr 2) * 5) - 3;
  YMonth := Temp div 153;
  If YMonth >= 10 then begin
    Inc(yYear);
    Dec(yMonth,12);
  End;
  inc(YMonth, 3);
  YDay := Temp mod 153;
  YDay := (YDay + 5) div 5;
  Year := YYear + (Julian * 100);
  Month := YMonth;
  Day := YDay;
End;

Procedure UnPackUnixTime(uts: longint; Var dt: datetime);
{ Unpacks a unix time stamp into a datetime record }
Var
  Temp,j1970,jNow: longint;
begin
  with dt do begin
    { Days past since Jan 1st 1970 will be put in temp }
    temp   := uts div 86400;
    { Figure out the Julian at Jan 1st 1970 }
    GregorianToJulian(1970,1,1,j1970);
    { Add the days }
    jNow := j1970 + temp;
    { Refigure it out }
    JulianToGregorian(jNow,year,month,day);

    { Seconds in the day will be put into temp }
    temp   := uts mod 86400;
    hour   := temp div 3600;
    min    := (temp mod 3600) div 60;
    sec    := temp mod 60;
  end;
end;

Procedure PackUnixTime(dt: datetime; var uts: longint);
{ Packs a datetime record into a unix time stamp }
var
  Temp,jNow,j1970: longint;
begin
  with dt do begin
    { Figure out the Julian at Jan 1st 1970 }
    GregorianToJulian(1970,1,1,j1970);
    { Figure out the Julian for the date }
    GregorianToJulian(year,month,day,jNow);
    { Figure out how many days we're talking about, multiple by seconds in }
    { a day, and presto }
    uts := (jNow-j1970)*86400;

    { Deal with hour, min, and sec }
    temp := longint(hour)*3600+longint(min)*60+longint(sec);
    uts := uts+temp;
  end;
end;

Function DayLightSavings(dt: datetime): boolean;
{ Returns if we're in daylight savings time which is from 2:00am on
  the first Sunday of April up to, but not including 2:00am on the
  last Sunday in October }
Var
  AprilSunday,OctoberSunday: datetime;
  PackedDt,
  D1,D2: longint;
Begin
  PackTime(dt,PackedDt);

  With AprilSunday do begin
    Day := DayOfWeek(01,04,Year);
    if Day>0 then day := (1+7)-Day else day := 1;
    month := 4; year := dt.year; hour := 2; min := 0; sec := 0;
  end;

  with OctoberSunday do begin
    Day := DayOfWeek(25,10,Year);
    if Day>0 then Day := (25+7)-Day else Day := 25;
    month := 10; year := dt.year; hour := 2; min := 0; sec := 0;
  end;

  PackTime(AprilSunday,d1);
  PackTime(OctoberSunday,d2);
  DayLightSavings := (PackedDt >= d1) and (PackedDt < d2);
End;

Function LeapYear(year: integer): boolean;
{ A leap year is a year evenly divisible by 4, and not be evenly divisible
  by 100 and not divisible by 400. }
begin
  LeapYear := (Year mod 4 = 0)
    and not ((year mod 100=0) and (year mod 400<>0));
end;

Function DayOfWeek(Year,Month,Day: Integer): Byte;
{ Figures out which day of the week it would have been on year, month, day }
var AddVal: shortint;
begin
 if Month < 3 then AddVal := 1 else AddVal := 0;
 DayOfWeek :=
  (
   3*year-
   (7*(year+(month+9) div 12)) div 4+
   (23*month) div 9+day+2+
   ((year-AddVal) div 100+1)*3 div 4-16+1
  ) mod 7;
end;

Function AddBackSlash(st: PathStr): PathStr;
{ Adds a backslash to the string "st" if it already does not have one }
Begin
  If st[length(st)] <> '\' then St := St + '\';
  AddBackSlash := st;
End;

Function RemoveBackSlash(st: PathStr): PathStr;
{ Removes trailing backslashes. Will not remove one from say C:\ }
var colonslash: boolean;
Begin
  colonslash := pos(':\',st)<>0;
  While (St[length(st)] = '\') do Delete(St,length(st),1);
  If colonslash and (pos(':\',st)=0) then st := addbackslash(st);
  RemoveBackSlash := st;
End;

Function BinToHunk(st: string): string;
{ Turns binary string into a MIME type encoded string }
var
  temp: string;
  i,j,times: byte;
begin
  temp := '';
  { Figures out how many times the hunking loop with need to be run }
  times := (length(st) div 3)*3;
  i := 0;
  if times <> 0 then
    repeat
      temp:=temp+char(byte(st[1+i]) shr 2);
      temp:=temp+char(((byte(st[1+i]) shl 4)+(byte(st[2+i]) shr 4)) and $3F);
      temp:=temp+char(((byte(st[2+i]) shl 2)+(byte(st[3+i]) shr 6)) and $3F);
      temp:=temp+char(byte(st[3+i]) and $3F);
      inc(i,3);
    until i = times;

  { Do the leftovers }
  case length(st) mod 3 of
    1: begin
      temp:=temp+char(byte(st[1+i]) shr 2);
      temp:=temp+char(((byte(st[1+i]) shl 4)) and $3F);
    end;
    2: begin
      temp:=temp+char(byte(st[1+i]) shr 2);
      temp:=temp+char(((byte(st[1+i]) shl 4)+(byte(st[2+i]) shr 4)) and $3F);
      temp:=temp+char(((byte(st[2+i]) shl 2)) and $3F);
    end;
  end;
  { Map it from the $00..$3F characters to something like PEM encoding }
  for j := 1 to length(temp) do case temp[j] of
    #0..#25: temp[j] := char(byte(temp[j])+byte('A'));
    #26..#51: temp[j] := char(byte(temp[j])-26+byte('a'));
    #52..#61: temp[j] := char(byte(temp[j])-52+byte('0'));
    #62: temp[j] := '+';
    #63: temp[j] := '/';
  end;
  BinToHunk := temp;
end;

Function HunkToBin(st: string): string;
{ Turns a MIME type encoded string into it's binary form }
var
  j,i,times: byte;
  temp: string;
begin
  { Demap from the MIME type encoding to the $00..$3F characters }
  for j := 1 to length(st) do case st[j] of
    'A'..'Z': st[j] := char(byte(st[j])-byte('A'));    { 0-25  }
    'a'..'z': st[j] := char(byte(st[j])-byte('a')+26); { 26-51 }
    '0'..'9': st[j] := char(byte(st[j])-byte('0')+52); { 52..61 }
    '+': st[j] := #62;                                 { 62 }
    '/': st[j] := #63;                                 { 63 }
  end;

  { Demap the $00..$3F codes to $00-$FF characters }
  times := (length(st) div 4)*4;
  temp := '';
  if times <> 0 then begin
    i := 0;
    repeat
      temp:=temp+char((byte(st[1+i]) shl 2)+(byte(st[2+i]) shr 4));
      temp:=temp+char((byte(st[2+i]) shl 4)+(byte(st[3+i]) shr 2));
      temp:=temp+char((byte(st[3+i]) shl 6)+(byte(st[4+i])));
      inc(i,4);
    until i = times;
  end;
  { Do the left overs }
  case length(st) mod 4 of
    1: temp:=temp+char((byte(st[1+i]) shl 2));
    2: temp:=temp+char((byte(st[1+i]) shl 2)+(byte(st[2+i]) shr 4));
    3: begin
      temp:=temp+char((byte(st[1+i]) shl 2)+(byte(st[2+i]) shr 4));
      temp:=temp+char((byte(st[2+i]) shl 4)+(byte(st[3+i]) shr 2));
    end;
  end;
  hunkTobin := temp;
end;

End.
